package ru.yandex.solomon.alert.canon;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.rule.AlertRule;
import ru.yandex.solomon.alert.rule.AlertRuleDeadlines;
import ru.yandex.solomon.alert.rule.AlertRuleFactory;
import ru.yandex.solomon.alert.rule.AlertRuleFactoryImpl;
import ru.yandex.solomon.alert.rule.AlertRuleFairDeadlines;
import ru.yandex.solomon.alert.rule.EmptyTemplateProcessor;
import ru.yandex.solomon.alert.rule.ExplainResult;
import ru.yandex.solomon.alert.rule.UnrollDeadlines;
import ru.yandex.solomon.alert.rule.usage.ProjectAlertRuleMetrics;
import ru.yandex.solomon.alert.template.MustacheTemplateFactory;
import ru.yandex.solomon.alert.template.TemplateFactory;
import ru.yandex.solomon.alert.unroll.MultiAlertUnrollFactory;
import ru.yandex.solomon.alert.unroll.MultiAlertUnrollFactoryImpl;
import ru.yandex.solomon.alert.unroll.MultiAlertUtils;
import ru.yandex.solomon.flags.FeatureFlagHolderStub;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metrics.client.MetricsClient;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.misc.ExceptionUtils.getStackTrace;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class Explainer {
    private final MultiAlertUnrollFactory unrollFactory;
    private final AlertRuleFactory ruleFactory;

    public Explainer(MetricsClient metricsClient) {
        TemplateFactory templateFactory = new MustacheTemplateFactory();

        unrollFactory = new MultiAlertUnrollFactoryImpl(metricsClient);
        var featureFlags = new FeatureFlagHolderStub();
        ruleFactory = new AlertRuleFactoryImpl(metricsClient, new ProjectAlertRuleMetrics(), templateFactory,
                () -> new ShardKey("solomon", "production", "alerting_statuses"), featureFlags);
    }

    public CompletableFuture<Alert> unrollIfNecessary(Alert alert) {
        if (!MultiAlertUnrollFactory.isSupportUnrolling(alert)) {
            return completedFuture(alert);
        }

        AlertRuleDeadlines deadlines = UnrollDeadlines.of(Instant.now(), 1, TimeUnit.MINUTES);

        return unrollFactory.create(alert)
                .unroll(deadlines)
                .thenApply(groupKeys -> {
                    if (groupKeys.labels.isEmpty()) {
                        return null;
                    }

                    Labels labels = groupKeys.labels.iterator().next();
                    return SubAlert.newBuilder()
                            .setId(MultiAlertUtils.getAlertId(alert, labels))
                            .setParent(alert)
                            .setGroupKey(labels)
                            .build();
                });
    }

    private CompletableFuture<ExplainResult> explain(Alert alert, Instant when, AlertRuleDeadlines deadlines) {
        AlertRule rule = ruleFactory.createAlertRule(alert);
        return rule.explain(when, deadlines);
    }

    public CompletableFuture<ExplainResult> explain(Alert alertOrSubAlert, Instant when) {
        AlertRuleDeadlines deadlines = AlertRuleFairDeadlines.ignoreLag(Instant.now(), 1, TimeUnit.MINUTES);

        return CompletableFutures.safeCall(() -> explain(alertOrSubAlert, when, deadlines))
                .handle((er, t) -> {
                    if (t != null) {
                        return EmptyTemplateProcessor.I.processTemplate(when, EvaluationStatus.ERROR.withDescription(getStackTrace(t)));
                    }
                    return er;
                });
    }
}
