package ru.yandex.solomon.alert.canon;

import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Assert;

import ru.yandex.misc.io.InputStreamSourceUtils2;
import ru.yandex.monlib.metrics.labels.validate.InvalidLabelException;
import ru.yandex.monlib.metrics.labels.validate.LabelsValidator;
import ru.yandex.solomon.alert.api.converters.AlertConverter;
import ru.yandex.solomon.alert.canon.protobuf.Converter;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertKey;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.domain.threshold.ThresholdAlert;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.TSubAlert;
import ru.yandex.solomon.alert.rule.ExplainResult;
import ru.yandex.solomon.alerting.canon.protobuf.TAlertEvaluationRecord;
import ru.yandex.solomon.alerting.canon.protobuf.TExplainResult;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TMetricsClientCapture;

import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class ReplayerCanonSupport {
    static final String METRICS_CLIENT_CAPTURE = "capture.pb.gz";
    static final String CANONICAL_EVALUATION_RESULT = "evaluationResults.pb.gz";

    static {
        try {
            LabelsValidator.checkKeyValid("foo-bar");
        } catch (InvalidLabelException e) {
            Assert.fail("Run with -Dru.yandex.solomon.LabelValidator=skip");
        }
    }

    /**
     * @param prefix Either local path as /foo/bar/baz/ or classpath:
     */
    public static TMetricsClientCapture readMetricsClientCapture(String prefix) throws IOException {
        // If you get resource not found exception
        // 1. ya make -t canon-data
        // 2. Regenerate project
        //
        // https://st.yandex-team.ru/DEVTOOLS-5511
        // https://ml.yandex-team.ru/thread/devtools/169166461003175860/

        return Serialization.readMetricsClientCapture(InputStreamSourceUtils2.valueOf(prefix + METRICS_CLIENT_CAPTURE).getInput());
    }

    public static Serialization.CanonResults readCanonicalResults(String prefix) throws IOException {
        return Serialization.readCanonicalResults(InputStreamSourceUtils2.valueOf(prefix + CANONICAL_EVALUATION_RESULT).getInput());
    }

    public static AlertKey keyFromRecord(TAlertEvaluationRecord record) {
        return record.getAlertOrSubAlertCase() == TAlertEvaluationRecord.AlertOrSubAlertCase.ALERT
                ? keyFromAlert(record.getAlert())
                : keyFromSubAlert(record.getSubAlert());
    }

    private static AlertKey keyFromSubAlert(TSubAlert subAlert) {
        return new AlertKey(subAlert.getProjectId(), subAlert.getParent().getId(), subAlert.getId());
    }

    private static AlertKey keyFromAlert(TAlert alert) {
        return new AlertKey(alert.getProjectId(), "", alert.getId());
    }

    public static TExplainResult evaluate(TAlertEvaluationRecord record, Explainer explainer, Instant when) {
        try {
            var alert = extractAlert(record);
            return evaluate(alert, explainer, when);
        } catch (Exception e) {
            return TExplainResult.newBuilder()
                    .setEvaluationStatus(TEvaluationStatus.newBuilder()
                            .setCode(TEvaluationStatus.ECode.ERROR)
                            .setDesctiption(e.getClass().getSimpleName() + ": " + e.getMessage()))
                    .build();
        }
    }

    public static TExplainResult evaluate(Alert alert, Explainer explainer, Instant when) {
        ExplainResult explainResult = join(explainer.explain(alert, when));
        return Converter.toProto(explainResult);
    }

    public static Alert extractAlert(TAlertEvaluationRecord record) {
        return record.getAlertOrSubAlertCase() == TAlertEvaluationRecord.AlertOrSubAlertCase.ALERT
                ? AlertConverter.protoToAlert(record.getAlert())
                : AlertConverter.protoToSubAlert(record.getSubAlert());
    }

    public static Set<String> computeFlappingAnnotationKeys(Alert alert) {
        Set<String> skippedAnnotations = new HashSet<>();
        if (isThreshold(alert)) {
            List<Pattern> groupByLabels = fixedLabels(alert).stream()
                    .map(label -> Pattern.compile("\\{\\{\\s*labels\\." + label + "\\s*}}"))
                    .collect(Collectors.toList());
            for (var entry : alert.getAnnotations().entrySet()) {
                if (containsFlappingMustache(entry.getValue(), groupByLabels)) {
                    skippedAnnotations.add(entry.getKey());
                }
            }
        }
        return skippedAnnotations;
    }

    private static final Pattern POINT_VALUE_ANNOTATION = Pattern.compile("\\{\\{\\s*pointValue\\s*}}");
    private static final Pattern POINT_TIME_ANNOTATION = Pattern.compile("\\{\\{\\s*pointTime\\s*}}");
    private static final Pattern OTHER_LABELS = Pattern.compile("\\{\\{\\s*labels\\.");

    private static boolean containsFlappingMustache(String value, List<Pattern> groupByLabels) {
        // groupByLabels are solid annotations, no flaps are possible
        for (Pattern groupByLabel : groupByLabels) {
            value = groupByLabel.matcher(value).replaceAll("");
        }
        return OTHER_LABELS.matcher(value).find() ||
                POINT_VALUE_ANNOTATION.matcher(value).find() ||
                POINT_TIME_ANNOTATION.matcher(value).find();
    }

    private static boolean isThreshold(Alert alert) {
        if (alert.getAlertType() == AlertType.SUB_ALERT) {
            return isThreshold(((SubAlert) alert).getParent());
        }
        return alert.getAlertType() == AlertType.THRESHOLD;
    }

    private static List<String> fixedLabels(Alert alert) {
        if (alert.getAlertType() == AlertType.SUB_ALERT) {
            return fixedLabels(((SubAlert) alert).getParent());
        }
        ThresholdAlert thresholdAlert = (ThresholdAlert) alert;
        ArrayList<String> fixedLabels = new ArrayList<>(thresholdAlert.getGroupByLabels());
        Selectors selectors = thresholdAlert.getSelectors();
        for (Selector s : selectors) {
            if (s.isExact()) {
                fixedLabels.add(s.getKey());
            }
        }
        return fixedLabels;
    }
}
