package ru.yandex.solomon.alert.canon;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.primitives.Doubles;
import org.junit.Assert;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Either;
import ru.yandex.salmon.proto.StockpileCanonicalProto.Archive;
import ru.yandex.solomon.alert.canon.protobuf.ArchiveConverterImpl;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alerting.canon.protobuf.TAlertTimeSeries;
import ru.yandex.solomon.alerting.canon.protobuf.TExplainResult;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class ResultMatcher {
    private static final Logger logger = LoggerFactory.getLogger(ResultMatcher.class);

    private final boolean strictSeriesCheck;
    private final boolean checkLabels;

    private ResultMatcher(Builder builder) {
        this.strictSeriesCheck = builder.strictSeriesCheck;
        this.checkLabels = builder.checkLabels;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {
        private boolean strictSeriesCheck = true;
        private boolean checkLabels = true;

        public Builder strictSeriesCheck(boolean strictSeriesCheck) {
            this.strictSeriesCheck = strictSeriesCheck;
            return this;
        }

        public Builder checkLabels(boolean checkLabels) {
            this.checkLabels = checkLabels;
            return this;
        }

        public ResultMatcher build() {
            return new ResultMatcher(this);
        }
    }

    public void assertEqualsIgnoringStackTrace(TExplainResult expected, TExplainResult actual, Set<String> skippedAnnotations) {
        assertAnnotationsEqualsWithoutTrace(expected.getEvaluationStatus(), actual.getEvaluationStatus(), skippedAnnotations);

        Assert.assertEquals("Series count", expected.getSeriesCount(), actual.getSeriesCount());

        var actualSeries = actual.getSeriesList().stream()
                .sorted(Comparator.comparing(TAlertTimeSeries::hashCode))
                .collect(Collectors.toList());

        var expectedSeries = actual.getSeriesList().stream()
                .sorted(Comparator.comparing(TAlertTimeSeries::hashCode))
                .collect(Collectors.toList());

        for (int i = 0; i < expected.getSeriesCount(); i++) {
            assertEquals("Series[" + i + "]", expectedSeries.get(i), actualSeries.get(i));
        }
    }

    private void assertEquals(String s, TAlertTimeSeries expected, TAlertTimeSeries actual) {
        Assert.assertEquals(s + ".Alias", expected.getAlias(), actual.getAlias());
        if (checkLabels) {
            Assert.assertEquals(s + ".Labels", expected.getLabels(), actual.getLabels());
        }
        assertEquals(s + ".Source", expected.getSource(), actual.getSource());
    }

    private static void assertApproxEquals(String msg, double expected, double actual, double relError) {
        double fuzz = relError * (Math.abs(expected) + Math.abs(actual));
        Assert.assertEquals(msg, expected, actual, fuzz);
    }

    private void assertEquals(String s, Archive expected, Archive actual) {
        if (strictSeriesCheck) {
            Assert.assertEquals(s + ".Header", expected.getHeader(), actual.getHeader());
            Assert.assertEquals(s + ".Columns", expected.getColumnsList(), actual.getColumnsList());
            Assert.assertEquals(s + ".Records", expected.getRecordsList(), actual.getRecordsList());
        } else {
            var expectedGd = ArchiveConverterImpl.I.fromProto(expected)
                    .toAggrGraphDataArrayList()
                    .toGraphDataShort(expected.getHeader().getType());
            var actualGd = ArchiveConverterImpl.I.fromProto(actual)
                    .toAggrGraphDataArrayList()
                    .toGraphDataShort(actual.getHeader().getType());
            Assert.assertEquals(expectedGd.getTimestamps(), actualGd.getTimestamps());
            var ts = expectedGd.getTimestamps();
            var expectedVals = expectedGd.getValues();
            var actualVals = actualGd.getValues();
            for (int i = 0; i < ts.length(); i++) {
                assertApproxEquals(s + ": At i = " + i + ", ts = " + ts.at(i),
                                expectedVals.at(i), actualVals.at(i), 1e-10);
            }
        }
    }

    private void assertAnnotationsEqualsWithoutTrace(
            TEvaluationStatus expected,
            TEvaluationStatus actual,
            Set<String> skippedAnnotations)
    {
        if (expected.getCode() == actual.getCode()) {
            logger.info("Alert status: " + expected.getCode());
        } else {
            logger.info("Expected: {} {}\n", expected.getCode(), expected.getAnnotationsMap().get("description"));
            logger.info("Actual  : {} {}\n", actual.getCode(), actual.getAnnotationsMap().get("description"));
        }

        Assert.assertEquals("EvaluationStatus.Code", expected.getCode(), actual.getCode());

        assertAnnotationsEqualsWithoutTrace(
                expected.getAnnotationsMap(),
                actual.getAnnotationsMap(),
                skippedAnnotations);
    }

    private void assertAnnotationsEqualsWithoutTrace(
            Map<String, String> expected,
            Map<String, String> actual,
            Set<String> skippedAnnotations)
    {
        Assert.assertTrue("Annotation keys", actual.keySet().containsAll(expected.keySet()));

        for (var key : expected.keySet()) {
            if (skippedAnnotations.contains(key)) {
                logger.warn("Annotation " + key + " was skipped due to possible flaps");
            } else {
                assertAnnotationEquals(key, expected.get(key), actual.get(key));
            }
        }
    }

    List<Either<String, Double>> parseNumbers(String[] tokens) {
        return Arrays.stream(tokens)
                .map(token -> {
                    Double value = Doubles.tryParse(token);
                    if (value == null) {
                        return Either.<String, Double>left(token);
                    } else {
                        return Either.<String, Double>right(value);
                    }
                })
                .collect(Collectors.toList());
    }

    private void assertAnnotationEquals(String key, String expected, String actual) {
        if (key.equals("causedBy") || key.equals("description")) {
            assertEqualsWithoutTrace("Annotation " + key, expected, actual);
        } else {
            // Allow fuzz when numbers are inlined to annotations
            var expectedTokens = expected.split("\\s+");
            var actualTokens = actual.split("\\s+");
            if (expectedTokens.length != actualTokens.length) {
                Assert.assertEquals("Annotation " + key, expected, actual);
            } else {
                List<Either<String, Double>> expectedParts = parseNumbers(expectedTokens);
                List<Either<String, Double>> actualParts = parseNumbers(actualTokens);

                for (int i = 0; i < expectedParts.size(); i++) {
                    var expectedPart = expectedParts.get(i);
                    var actualPart = actualParts.get(i);

                    if (expectedPart.isRight() && actualPart.isRight()) {
                        assertApproxEquals("Annotation " + key + " (token " + i + ")",
                                expectedPart.getRight(), actualPart.getRight(), 1e-10);
                    } else if (expectedPart.isLeft() && actualPart.isLeft()) {
                        Assert.assertEquals("Annotation " + key + " (token " + i + ")",
                                expectedPart.getLeft(), actualPart.getLeft());
                    } else {
                        Assert.assertEquals("Annotation " + key + " (token " + i + ")",
                                expectedPart, actualPart);
                    }
                }
            }
        }
    }

    private final static String TRACEBACK_MARKER = "\n\tat ru.yandex.solomon";
    private final static String EXCEPTION_MARKER = "ru.yandex.solomon.expression.exceptions.";
    private final static String AGGREGATION_ERROR = ": Not able to apply function ";
    private final static String RESOLUTION_ERROR = " was resolved into multiple lines while only one line was allowed: ";

    private String skipMaybeException(String message) {
        if (!message.startsWith(EXCEPTION_MARKER)) {
            return message;
        }
        return message.substring(message.indexOf(":") + 2);
    }

    private String truncateMaybeAggregationError(String message) {
        if (message.contains(AGGREGATION_ERROR)) {
            return message.substring(0, message.indexOf("["));
        }

        if (message.contains(RESOLUTION_ERROR)) {
            return message.substring(0, message.indexOf(RESOLUTION_ERROR) + RESOLUTION_ERROR.length());
        }

        return message;
    }

    private void assertEqualsWithoutTrace(String key, String expected, String actual) {
        int posExpected = expected.indexOf(TRACEBACK_MARKER);
        if (posExpected != -1) {
            expected = expected.substring(0, posExpected);
        }
        expected = skipMaybeException(expected);
        expected = truncateMaybeAggregationError(expected);

        int posActual = actual.indexOf(TRACEBACK_MARKER);
        if (posActual != -1) {
            actual = actual.substring(0, posActual);
        }
        actual = skipMaybeException(actual);
        actual = truncateMaybeAggregationError(actual);

        assertEqualsWithoutActualNoDataSelectors(key, expected, actual);
    }

    private final static String NO_METRICS_FOUND_MARKER = "No metrics found by selectors: {";
    private final static String NO_METRICS_FOUND_MARKER2 = "No metrics found by selector: {";
    private final static String NO_METRICS_FOUND_MARKER3 = "No metrics by selectors {";
    private final static String NO_POINTS_IN_METRIC_MARKER = "No points in metric {";
    private final static String NO_POINTS_AT_METRIC_MARKER = "No points at metric: {";

    private static boolean noPoints(String message) {
        return message.startsWith(NO_POINTS_IN_METRIC_MARKER) || message.startsWith(NO_POINTS_AT_METRIC_MARKER);
    }

    private static boolean noMetrics(String message) {
        return message.startsWith(NO_METRICS_FOUND_MARKER)
                || message.startsWith(NO_METRICS_FOUND_MARKER2)
                || message.startsWith(NO_METRICS_FOUND_MARKER3);
    }

    private void assertEqualsWithoutActualNoDataSelectors(String key, String expected, String actual) {
        if (noMetrics(expected) && noMetrics(actual)) {
            return;
        }
        if (noPoints(expected) && noPoints(actual)) {
            return;
        }
        Assert.assertEquals(key + " (truncated)", expected, actual);
    }
}
