package ru.yandex.solomon.alert.canon;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import ru.yandex.solomon.alerting.canon.protobuf.TAlertEvaluationRecord;
import ru.yandex.solomon.config.protobuf.Time;
import ru.yandex.solomon.config.protobuf.TimeUnit;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TMetricsClientCapture;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class Serialization {
    public static class CanonResults {
        public Instant evaluationMoment;
        public List<TAlertEvaluationRecord> alertEvaluationRecords;
    }

    public static TMetricsClientCapture readMetricsClientCapture(InputStream compressedInputStream) throws IOException {
        return TMetricsClientCapture.parseFrom(new GZIPInputStream(compressedInputStream));
    }

    public static void writeMetricsClientCapture(
            TMetricsClientCapture capture,
            @WillNotClose OutputStream compressedOuputStream) throws IOException
    {
        GZIPOutputStream os = new GZIPOutputStream(compressedOuputStream);
        capture.writeTo(os);
        os.finish();
    }

    public static CanonResults readCanonicalResults(InputStream compressedInputStream) throws IOException {
        InputStream is = new GZIPInputStream(compressedInputStream);

        CanonResults results = new CanonResults();

        Time alertEvaluationMomentProto = Time.parseDelimitedFrom(is);
        results.evaluationMoment = Instant.ofEpochMilli(alertEvaluationMomentProto.getValue());
        TAlertEvaluationRecord record;
        results.alertEvaluationRecords = new ArrayList<>();
        while ((record = TAlertEvaluationRecord.parseDelimitedFrom(is)) != null) {
            results.alertEvaluationRecords.add(record);
        }

        return results;
    }

    public static void writeCanonicalResults(
            Instant evaluationMoment,
            List<TAlertEvaluationRecord> alertEvaluationRecords,
            @WillNotClose OutputStream compressedOuputStream) throws IOException
    {
        GZIPOutputStream os = new GZIPOutputStream(compressedOuputStream);

        Time.newBuilder()
                .setUnit(TimeUnit.MILLISECONDS)
                .setValue(evaluationMoment.toEpochMilli())
                .build()
                .writeDelimitedTo(os);

        for (var record : alertEvaluationRecords) {
            record.writeDelimitedTo(os);
        }
        os.finish();
    }

    public static void writeCanonicalResults(
            CanonResults canonResults,
            @WillNotClose OutputStream compressedOutputStream) throws IOException
    {
        writeCanonicalResults(
                canonResults.evaluationMoment,
                canonResults.alertEvaluationRecords,
                compressedOutputStream);
    }
}
