package ru.yandex.solomon.coremon;

import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.resourcemanager.GrpcResourceManagerClient;
import ru.yandex.cloud.resourcemanager.ResourceManagerClient;
import ru.yandex.cloud.resourcemanager.ResourceManagerClientOptions;
import ru.yandex.cloud.token.IamTokenClient;
import ru.yandex.cloud.token.Jwt;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.grpc.conf.GrpcConfigurationContext;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolver;
import ru.yandex.solomon.cloud.resource.resolver.CloudByFolderResolverImpl;
import ru.yandex.solomon.cloud.resource.resolver.FolderResolver;
import ru.yandex.solomon.cloud.resource.resolver.FolderResolverImpl;
import ru.yandex.solomon.config.protobuf.coremon.TCoremonCloudConfig;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.coremon.meta.service.cloud.NameResolverResourceFinder;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceResolver;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceResolverImpl;
import ru.yandex.solomon.coremon.meta.service.cloud.ResourceFinder;
import ru.yandex.solomon.ctx.ServiceAuthContext;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.name.resolver.client.NameResolverClient;
import ru.yandex.solomon.name.resolver.client.NoopNameResolverClient;
import ru.yandex.solomon.name.resolver.client.grpc.GrpcNameResolverClient;
import ru.yandex.solomon.spring.ConditionalOnBean;
import ru.yandex.solomon.util.SolomonEnv;

/**
 * @author Sergey Polovko
 */
@Configuration
@ConditionalOnBean(TCoremonCloudConfig.class)
@Import({
        GrpcConfigurationContext.class,
        ServiceAuthContext.class

})
public class CoremonCloudContext {

    private static final Logger logger = LoggerFactory.getLogger(CoremonCloudContext.class);

    private static final Duration DEFAULT_CONNECT_TIMEOUT = Duration.ofSeconds(5);
    private static final Duration DEFAULT_REQUEST_TIMEOUT = Duration.ofSeconds(30);

    private final TCoremonCloudConfig cloudConfig;
    private final TAuthConfig authConfig;
    private final ThreadPoolProvider threads;

    public CoremonCloudContext(TCoremonCloudConfig cloudConfig, TAuthConfig authConfig, ThreadPoolProvider threads) {
        this.cloudConfig = cloudConfig;
        this.threads = threads;
        this.authConfig = authConfig;
    }

    @Bean
    NameResolverClient nameResolverClient(
            @Qualifier("ClientId") String clientId,
            MetricRegistry registry,
            ClientOptionsFactory clientOptionsFactory)
    {
        var config = cloudConfig.getNameResolverClient().getGrpcConfig();
        if (config.equals(TGrpcClientConfig.getDefaultInstance())) {
            return new NoopNameResolverClient();
        }

        var opts = clientOptionsFactory.newBuilder(
                "CloudConfig.NameResolverClient",
                config)
            .setClientId(clientId)
            .setMetricRegistry(registry)
            .build();
        return new GrpcNameResolverClient(config.getAddressesList(), opts);
    }

    @Bean
    ResourceFinder resourceFinder(NameResolverClient client) {
        return new NameResolverResourceFinder(client);
    }

    @Bean
    ReferenceResolver referenceResolver(FeatureFlagsHolder flags, SolomonConfHolder confs) {
        return new ReferenceResolverImpl(flags, confs);
    }

    @Bean(name = "iamTokenProvider")
    TokenProvider tokenProvider(IamTokenClient iamTokenClient, TAuthConfig authConfig, ThreadPoolProvider threads) {
        if (!authConfig.hasIamConfig()) {
            logger.warn("using fake token provider: no iam config");
            return TokenProvider.of("fake-token");
        }

        if (!authConfig.getIamConfig().hasKeyAuth()) {
            logger.warn("using fake token provider: no key auth");
            return TokenProvider.of("fake-token");
        }

        var keyAuth = authConfig.getIamConfig().getKeyAuth();

        var keyFilePath = Path.of(keyAuth.getKeyPath());
        if (SolomonEnv.DEVELOPMENT.isActive() && !Files.exists(keyFilePath)) {
            logger.warn("using fake token provider: development env");
            return TokenProvider.of("fake-token");
        }

        var jwtBuilder = Jwt.newBuilder()
            .withAccountId(keyAuth.getAccountId())
            .withKeyId(keyAuth.getKeyId())
            .withPrivateKey(keyFilePath)
            .withTtl(Duration.ofHours(1));

        return TokenProvider.iam(iamTokenClient, jwtBuilder, threads.getSchedulerExecutorService());
    }

    @Bean
    ResourceManagerClient resourceManagerClient(
        TCoremonCloudConfig cloudConfig,
        @Qualifier("iamTokenProvider") TokenProvider tokenProvider,
        ThreadPoolProvider threads,
        MetricRegistry registry)
    {
        var config = cloudConfig.getResourceManagerClient();
        var executor = threads.getExecutorService(
            config.getThreadPoolName(),
            "CloudConfig.ResourceManagerClient.ThreadPoolName");

        return new GrpcResourceManagerClient(
            ResourceManagerClientOptions.forAddress(config.getHost(), config.getPort())
                .withUserAgent("SolomonCoremon")
                .withConnectTimeout(DEFAULT_CONNECT_TIMEOUT)
                .withRequestTimeout(DEFAULT_REQUEST_TIMEOUT)
                .withHandlerExecutor(executor)
                .withTokenProvider(tokenProvider)
                .withRegistry(registry)
                .withResolveExistingOnly(config.getResolveExistingOnly()));
    }

    @Bean
    CloudByFolderResolver cloudByFolderResolver(ResourceManagerClient resourceManagerClient) {
        return new CloudByFolderResolverImpl(resourceManagerClient);
    }

    @Bean
    FolderResolver folderResolver(
        SolomonConfHolder confHolder,
        CloudByFolderResolver cloudByFolderResolver)
    {
        return new FolderResolverImpl(confHolder, cloudByFolderResolver);
    }

}
