package ru.yandex.solomon.coremon;

import java.time.Clock;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.TKikimrClientConfig;
import ru.yandex.solomon.config.protobuf.TaskSchedulerConfig;
import ru.yandex.solomon.config.protobuf.coremon.DeleteMetricsConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.coremon.meta.db.DeletedMetricsDao;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsCheckTaskHandler;
import ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsMoveTaskHandler;
import ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRollbackTaskHandler;
import ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsTaskMetrics;
import ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsTerminateTaskHandler;
import ru.yandex.solomon.coremon.tasks.removeShard.RemoveShardTaskHandler;
import ru.yandex.solomon.locks.dao.LocksDao;
import ru.yandex.solomon.locks.dao.ydb.YdbLocksDao;
import ru.yandex.solomon.scheduler.context.TaskSchedulerContext;
import ru.yandex.solomon.selfmon.mon.DaoMetricsProxy;
import ru.yandex.solomon.spring.ConditionalOnBean;
import ru.yandex.stockpile.client.StockpileClient;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@ConditionalOnBean(TaskSchedulerConfig.class)
public class CoremonSchedulerContext extends TaskSchedulerContext {

    public CoremonSchedulerContext(
            @Qualifier("localTableClient") TableClient tableClient,
            @Qualifier("localSchemeClient") SchemeClient schemeClient,
            @Qualifier("KikimrClientConfig") TKikimrClientConfig ydbConfig,
            TaskSchedulerConfig schedulerConfig,
            ThreadPoolProvider threads)
    {
        super(ydbConfig.getSchemaRoot() + "/V1", schedulerConfig, threads, tableClient, schemeClient);
    }

    @Bean
    public RemoveShardTaskHandler removeShardTaskHandler(
            SolomonConfHolder confHolder,
            StockpileClient stockpile,
            MetricsDaoFactory metricsDao)
    {
        return new RemoveShardTaskHandler(confHolder, stockpile, metricsDao, executor(), timer());
    }

    @Bean
    public DeleteMetricsTaskMetrics deleteMetricsTaskMetrics(
        MetricRegistry metricRegistry,
        DeleteMetricsConfig config)
    {
        return new DeleteMetricsTaskMetrics(metricRegistry, config);
    }

    @Bean
    public DeleteMetricsCheckTaskHandler deleteMetricsCheckTaskHandler(
        SolomonConfHolder confHolder,
        StockpileClient stockpileClient,
        MetabaseShardResolver shardResolver,
        DeleteMetricsTaskMetrics metrics)
    {
        return new DeleteMetricsCheckTaskHandler(
            metrics,
            confHolder,
            stockpileClient,
            shardResolver,
            executor(),
            timer());
    }

    @Bean
    public DeleteMetricsMoveTaskHandler deleteMetricsMoveTaskHandler(
        @Qualifier("localTableClient") TableClient tableClient,
        @Qualifier("KikimrClientConfig") TKikimrClientConfig ydbConfig,
        MetricRegistry metricRegistry,
        SolomonConfHolder confHolder,
        StockpileClient stockpileClient,
        MetabaseShardResolver shardResolver,
        DeletedMetricsDao deletedMetricsDao,
        MetricsDaoFactory metricsDaoFactory,
        DeleteMetricsTaskMetrics metrics)
    {
        var path = ydbConfig.getSchemaRoot() + "/V1/DeletedMetricsLocks";
        var ydbLocksDao = new YdbLocksDao(tableClient, Clock.systemUTC(), path);
        var locksDao = DaoMetricsProxy.of(ydbLocksDao, LocksDao.class, metricRegistry);
        ydbLocksDao.createSchema();

        return new DeleteMetricsMoveTaskHandler(
            metrics,
            confHolder,
            stockpileClient,
            shardResolver,
            locksDao,
            deletedMetricsDao,
            metricsDaoFactory,
            executor(),
            timer());
    }

    @Bean
    public DeleteMetricsRollbackTaskHandler deleteMetricsRollbackTaskHandler(
        SolomonConfHolder confHolder,
        StockpileClient stockpileClient,
        MetabaseShardResolver shardResolver,
        DeletedMetricsDao deletedMetricsDao,
        DeleteMetricsTaskMetrics metrics)
    {
        return new DeleteMetricsRollbackTaskHandler(
            metrics,
            confHolder,
            stockpileClient,
            shardResolver,
            deletedMetricsDao,
            executor(),
            timer());
    }

    @Bean
    public DeleteMetricsTerminateTaskHandler deleteMetricsTerminateTaskHandler(
        SolomonConfHolder confHolder,
        StockpileClient stockpileClient,
        MetabaseShardResolver shardResolver,
        DeletedMetricsDao deletedMetricsDao,
        DeleteMetricsTaskMetrics metrics)
    {
        return new DeleteMetricsTerminateTaskHandler(
            metrics,
            confHolder,
            stockpileClient,
            shardResolver,
            deletedMetricsDao,
            executor(),
            timer());
    }
}
