package ru.yandex.solomon.coremon;

import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import io.netty.buffer.ByteBuf;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monitoring.coremon.TDataProcessResponse;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.LabelsBuilder;
import ru.yandex.solomon.core.urlStatus.UrlStatusTypes;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.push.LastPushedData;
import ru.yandex.solomon.coremon.stockpile.CoremonShardStockpile;
import ru.yandex.solomon.coremon.stockpile.TwoResponsesRequest;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.memory.layout.MemInfoProvider;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;
import ru.yandex.solomon.model.protobuf.MetricFormat;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.manager.find.NamedObject;
import ru.yandex.solomon.staffOnly.manager.special.InstantMillis;
import ru.yandex.solomon.staffOnly.manager.special.PullHere;
import ru.yandex.solomon.staffOnly.manager.table.TableColumn;


/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class CoremonShard implements NamedObject, MemInfoProvider, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(CoremonShard.class);

    @PullHere
    private final CoremonShardStockpile processingShard;

    @PullHere
    private final MetabaseShard metabaseShard;

    private final AtomicLong requestStarted = new AtomicLong();
    private final AtomicLong requestCompletedSuccess = new AtomicLong();
    private final AtomicLong requestCompletedError = new AtomicLong();

    private Throwable lastPushException;
    @InstantMillis
    private long lastPushExceptionInstant;

    // for PUSH debugging
    @Nullable
    private volatile LastPushedData lastPushedData;
    private final AtomicBoolean captureNextPush = new AtomicBoolean(false);

    public CoremonShard(CoremonShardStockpile processingShard, MetabaseShard metabaseShard) {
        this.processingShard = processingShard;
        this.metabaseShard = metabaseShard;
    }

    @ManagerMethod
    void captureNextPushData() {
        captureNextPush.set(true);
    }

    @ManagerMethod
    void clearLastPushData() {
        LastPushedData lastPushedData = this.lastPushedData;
        this.lastPushedData = null;
        if (lastPushedData != null) {
            lastPushedData.clear();
        }
    }

    public void stop() {
        processingShard.stop();
        metabaseShard.stop();
    }

    public CoremonShardStockpile getProcessingShard() {
        return processingShard;
    }

    public MetabaseShard getMetabaseShard() {
        return metabaseShard;
    }

    @TableColumn
    public int inFlight() {
        return (int) (requestStarted.get() - requestCompletedSuccess.get() - requestCompletedError.get());
    }

    @Override
    public String namedObjectId() {
        return processingShard.namedObjectId();
    }

    public String getId() {
        return processingShard.getId();
    }

    public int getNumId() {
        return processingShard.getNumId();
    }

    @Override
    public MemoryBySubsystem memoryBySystem() {
        MemoryBySubsystem r = new MemoryBySubsystem();
        // TODO: memory by fetcher
        processingShard.addMemoryInfo(r);
        metabaseShard.addMemoryInfo(r);
        return r;
    }

    public boolean isReady() {
        return processingShard.isReady();
    }

    public LabelAllocator labelAllocator() {
        return processingShard.labelAllocator;
    }

    public CompletableFuture<TDataProcessResponse> push(
        String host, Labels hostOptLabels,
        MetricFormat metricFormat,
        long instantMillis, ByteBuf response,
        long prevInstantMillis, ByteBuf prevResponse,
        boolean isPush,
        boolean onlyNewFormatWrites)
    {
        Labels hostLabels = hostLabels(host, isPush);

        hostOptLabels = filterOptLabels(hostOptLabels, processingShard.getAggrUseLabelNames());

        if (captureNextPush.getAndSet(false)) {
            LastPushedData lastPushedData = this.lastPushedData;
            this.lastPushedData = new LastPushedData(
                host,
                hostOptLabels,
                metricFormat,
                response.retainedSlice(),
                instantMillis,
                isPush);
            if (lastPushedData != null) {
                lastPushedData.clear();
            }
        }

        TwoResponsesRequest batch = new TwoResponsesRequest(
            hostLabels, hostOptLabels, metricFormat,
            response, instantMillis,
            prevResponse, prevInstantMillis,
            isPush,
            onlyNewFormatWrites);

        var future = processingShard.pushPage(batch);
        requestStarted.incrementAndGet();
        return future.handle((result, t) -> {
            if (t != null) {
                requestCompletedError.incrementAndGet();

                Throwable cause = CompletableFutures.unwrapCompletionException(t);
                UrlStatusType statusType = UrlStatusTypes.classifyError(cause);

                if (statusType == UrlStatusType.UNKNOWN_ERROR) {
                    logger.warn("failed to push into " + getId(), cause);
                }

                lastPushException = cause;
                lastPushExceptionInstant = System.currentTimeMillis();

                return TDataProcessResponse.newBuilder()
                    .setStatus(statusType)
                    .setErrorMessage(Strings.nullToEmpty(cause.getMessage()))
                    .build();
            }

            requestCompletedSuccess.incrementAndGet();
            return TDataProcessResponse.newBuilder()
                .setStatus(result.getStatusType())
                .setErrorMessage(result.getErrorMessage())
                .setSuccessMetricCount(result.getMetricCount())
                .build();
        });
    }

    private Labels filterOptLabels(Labels labels, Set<String> aggrUseLabelNames) {
        if (labels.isEmpty()) {
            return labels;
        }

        if (aggrUseLabelNames.isEmpty()) {
            return Labels.empty();
        }

        LabelsBuilder builder = Labels.builder(labels.size());
        labels.forEach(label -> {
            if (aggrUseLabelNames.contains(label.getKey())) {
                builder.add(label);
            }
        });

        return builder.build();
    }

    private Labels hostLabels(String host, boolean isPush) {
        if (isPush || host.isEmpty()) {
            return Labels.of();
        }

        if (!processingShard.isHostRequired()) {
            host = LabelKeys.HOST;
        }

        return Labels.of(processingShard.labelAllocator.alloc(LabelKeys.HOST, host));
    }

    public void setOptsForMetabase(CoremonShardOpts opts) {
        this.metabaseShard.setMetricNameLabel(opts.getMetricNameLabel());
        this.metabaseShard.setOnlyNewFormatWrites(opts.isOnlyNewFormatWrites());
        this.metabaseShard.setOnlyNewFormatReads(opts.isOnlyNewFormatReads());
        this.metabaseShard.setServiceProvider(opts.getServiceProvider());
        this.metabaseShard.setShardState(opts.getShardState());
        this.metabaseShard.setMaxFileMetrics(opts.getQuota().getMaxFileMetrics());
    }

    @Override
    public void close() {
        processingShard.stop();
        metabaseShard.stop();
    }

    public CompletableFuture<Void> getLoadFuture() {
        return metabaseShard.getLoadFuture();
    }
}
