package ru.yandex.solomon.coremon;

import java.util.Arrays;
import java.util.concurrent.ExecutorService;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.solomon.config.protobuf.coremon.TMetabaseMetricNameMigrationConfig;
import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.conf.aggr.AggrRuleConf;
import ru.yandex.solomon.coremon.meta.db.MetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardImpl;
import ru.yandex.solomon.coremon.stockpile.CoremonMericConfDetailed;
import ru.yandex.solomon.coremon.stockpile.CoremonShardStockpile;
import ru.yandex.solomon.coremon.stockpile.MetricProcessorFactory;
import ru.yandex.solomon.coremon.stockpile.ProcessingWriter;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.metrics.parser.json.TreeParserJson;
import ru.yandex.stockpile.client.StockpileClient;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class CoremonShardFactory {

    private final ExecutorService parsingThreadPool;
    private final ExecutorService miscThreadPool;
    private final MetricsDaoFactory metricsDaoFactory;
    private final StockpileClient stockpileClient;
    private final MetricProcessorFactory processorFactory;
    private final TMetabaseMetricNameMigrationConfig metricNameMigrationConfig;
    private final ProcessingWriter stockpileBufferedWriter;

    public CoremonShardFactory(
        ExecutorService parsingThreadPool,
        ExecutorService miscThreadPool,
        MetricsDaoFactory metricsDaoFactory,
        StockpileClient stockpileClient,
        MetricProcessorFactory processorFactory,
        TMetabaseMetricNameMigrationConfig metricNameMigrationConfig,
        ProcessingWriter stockpileBufferedWriter)
    {
        this.parsingThreadPool = parsingThreadPool;
        this.miscThreadPool = miscThreadPool;
        this.metricsDaoFactory = metricsDaoFactory;
        this.stockpileClient = stockpileClient;
        this.processorFactory = processorFactory;
        this.metricNameMigrationConfig = metricNameMigrationConfig;
        this.stockpileBufferedWriter = stockpileBufferedWriter;
    }

    public CoremonShard createShard(ShardConfDetailed shardConf) {
        var opts = new CoremonShardOpts(shardConf, metricNameMigrationConfig);

        // each shard will have its own labels pool to reduce contention
        LabelAllocator labelAllocator = new InterningLabelAllocator();
        TreeParserJson treeParser = new TreeParserJson(labelAllocator, opts.getMetricNameLabel());

        MetabaseShard metabaseShard = new MetabaseShardImpl(
                opts.getId().getNumId(), opts.getId().getShardId(), opts.getFolderId(),
                opts.getNumPartitions(), opts.getId().getShardKey(),
                metricsDaoFactory, labelAllocator, this.miscThreadPool, stockpileClient);

        CoremonShardStockpile processingShard = new CoremonShardStockpile(
                newMetricConf(opts),
                treeParser,
                labelAllocator,
                parsingThreadPool,
                miscThreadPool,
                stockpileClient,
                processorFactory,
                opts,
                stockpileBufferedWriter,
                metabaseShard);

        var coremonShard = new CoremonShard(processingShard, metabaseShard);
        coremonShard.setOptsForMetabase(opts);
        return coremonShard;
    }

    public boolean updateShard(CoremonShard prev, ShardConfDetailed shardConf) {
        CoremonShardOpts nextOpts = new CoremonShardOpts(shardConf, metricNameMigrationConfig);
        CoremonShardOpts prevOpts = prev.getProcessingShard().getOpts();
        if (nextOpts.equals(prevOpts)) {
            return true;
        }

        if (prevOpts.reloadShard(nextOpts)) {
            return false;
        }

        prev.getProcessingShard().setOpts(nextOpts);
        prev.setOptsForMetabase(nextOpts);

        return true;
    }

    private CoremonMericConfDetailed newMetricConf(CoremonShardOpts opts) {
        AggrRuleConf[] aggrRules = Arrays.stream(opts.getMetricConf().getAggrRules())
                .map(rule -> new AggrRuleConf(rule, opts.getId().getShardKey()))
                .toArray(AggrRuleConf[]::new);
        boolean rawDataMemOnly = opts.getMetricConf().isRawDataMemOnly();
        return new CoremonMericConfDetailed(aggrRules, rawDataMemOnly);
    }
}
