package ru.yandex.solomon.coremon;

import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.solomon.config.protobuf.coremon.TMetabaseMetricNameMigrationConfig;
import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.conf.ShardKeyAndId;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.ServiceMetricConf;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.core.db.model.ValidationMode;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.stockpile.api.EDecimPolicy;


/**
 * @author alexlovkov
 **/
public class CoremonShardOpts extends DefaultObject {

    private final String folderId;
    private final ShardKeyAndId id;
    private final int numPartitions;
    private final long fetchMillis;
    private final long gridMillis;
    private final EDecimPolicy decimPolicyId;
    private final ServiceMetricConf metricConf;
    private final ValidationMode validationMode;
    private final ShardState shardState;
    private final String metricNameLabel;
    private final String serviceProvider;
    private final boolean onlyNewFormatReads;
    private final boolean onlyNewFormatWrites;
    private final CoremonShardQuota quota;

    CoremonShardOpts(ShardConfDetailed conf, TMetabaseMetricNameMigrationConfig metricNameMigrationConfig) {
        this.id = conf.getShardKeyAndId();
        this.numPartitions = conf.getNumPartitions();
        this.folderId = conf.getRaw().getFolderId();
        this.decimPolicyId = conf.getDecimPolicy().toProto();
        this.fetchMillis = TimeUnit.SECONDS.toMillis(conf.getInterval());
        this.gridMillis = conf.hasGrid()
                ? TimeUnit.SECONDS.toMillis(conf.getGridSec())
                : 0L;
        this.metricConf = conf.getMetricConf();
        this.validationMode = conf.getValidationMode();
        this.shardState = conf.getRaw().getState();
        this.metricNameLabel = computeMetricNameLabel(conf, metricNameMigrationConfig);
        this.onlyNewFormatReads = metricNameMigrationConfig.getOnlyNewFormatReads() || conf.getProject().isOnlyNewFormatReads();
        this.onlyNewFormatWrites = metricNameMigrationConfig.getOnlyNewFormatWrites() || conf.getProject().isOnlyNewFormatWrites();
        this.serviceProvider = Nullables.orEmpty(conf.getService().getRaw().getServiceProvider());
        this.quota = new CoremonShardQuota(
                conf.getMaxMetricsPerUrl(),
                conf.getMaxFileMetrics(),
                conf.getMaxMemMetrics());
    }

    public CoremonShardOpts(Builder builder) {
        this.id = builder.id;
        this.folderId = Nullables.orEmpty(builder.folderId);
        this.fetchMillis = builder.fetchMillis;
        this.gridMillis = builder.gridMillis == 0
                ? builder.fetchMillis
                : builder.gridMillis;
        this.decimPolicyId = Nullables.orDefault(builder.decimPolicy, DecimPolicy.DEFAULT).toProto();
        this.quota = Nullables.orDefault(builder.quota, CoremonShardQuota.DEFAULT);
        this.metricConf = builder.metricConf;
        this.validationMode = Nullables.orDefault(builder.validationMode, ValidationMode.STRICT_FAIL);
        this.shardState = builder.shardState;
        this.metricNameLabel = builder.metricNameLabel;
        this.serviceProvider = builder.serviceProvider;
        this.onlyNewFormatReads = builder.onlyNewFormatReads;
        this.onlyNewFormatWrites = builder.onlyNewFormatWrites;
        this.numPartitions = builder.numPartitions;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    private String computeMetricNameLabel(
        ShardConfDetailed conf,
        TMetabaseMetricNameMigrationConfig migrationConfig)
    {
        String metricNameLabel = conf.getMetricNameLabel();

        if (!metricNameLabel.isEmpty()) {
            return metricNameLabel;
        }

        if (conf.getProject().isOnlyMetricNameShards() || migrationConfig.getOnlyMetricNameShards()) {
            return StringUtils.defaultIfEmpty(migrationConfig.getDefaultMetricNameLabel(), LabelKeys.SENSOR);
        }

        return "";
    }

    boolean reloadShard(CoremonShardOpts other) {
        return !id.equals(other.id) || !metricConf.equals(other.metricConf);
    }

    public ShardKeyAndId getId() {
        return id;
    }

    public String getFolderId() {
        return folderId;
    }

    public int getNumPartitions() {
        return numPartitions;
    }

    public CoremonShardQuota getQuota() {
        return quota;
    }

    public long getFetchMillis() {
        return fetchMillis;
    }

    public long getGridMillis() {
        return gridMillis;
    }

    public ServiceMetricConf getMetricConf() {
        return metricConf;
    }

    public ValidationMode getValidationMode() {
        return validationMode;
    }

    public ShardState getShardState() {
        return shardState;
    }

    public String getMetricNameLabel() {
        return metricNameLabel;
    }

    public boolean isOnlyNewFormatReads() {
        return onlyNewFormatReads;
    }

    public boolean isOnlyNewFormatWrites() {
        return onlyNewFormatWrites;
    }

    public EDecimPolicy getDecimPolicyId() {
        return decimPolicyId;
    }

    public String getServiceProvider() {
        return serviceProvider;
    }

    public static final class Builder {
        private ShardKeyAndId id;
        private String folderId;
        private long fetchMillis;
        private long gridMillis;
        private DecimPolicy decimPolicy;
        private CoremonShardQuota quota;
        private ValidationMode validationMode;
        private int numPartitions;
        private ShardState shardState;
        private String metricNameLabel;
        private ServiceMetricConf metricConf;
        private boolean onlyNewFormatReads;
        private boolean onlyNewFormatWrites;
        private String serviceProvider;

        public Builder() {
        }

        public Builder(CoremonShardOpts opts) {
            this.id = opts.id;
            this.folderId = opts.folderId;
            this.fetchMillis = opts.fetchMillis;
            this.decimPolicy = DecimPolicy.fromProto(opts.decimPolicyId);
            this.quota = opts.quota;
            this.metricConf = opts.metricConf;
            this.validationMode = opts.validationMode;
            this.shardState = opts.shardState;
            this.metricNameLabel = opts.metricNameLabel;
            this.onlyNewFormatReads = opts.onlyNewFormatReads;
            this.onlyNewFormatWrites = opts.onlyNewFormatWrites;
            this.serviceProvider = opts.serviceProvider;
            this.numPartitions = opts.numPartitions;
        }

        public Builder withId(ShardKeyAndId id) {
            this.id = id;
            return this;
        }

        public Builder withServiceProvider(String serviceProvider) {
            this.serviceProvider = serviceProvider;
            return this;
        }

        public Builder withNumPartitions(int numPartitions) {
            this.numPartitions = numPartitions;
            return this;
        }

        public Builder withFolderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder withFetchMillis(long stepMillis) {
            this.fetchMillis = stepMillis;
            return this;
        }

        public Builder withGridMillis(long gridMillis) {
            this.gridMillis = gridMillis;
            return this;
        }

        public Builder withDecimPolicy(DecimPolicy decimPolicy) {
            this.decimPolicy = decimPolicy;
            return this;
        }

        public Builder withQuota(CoremonShardQuota quota) {
            this.quota = quota;
            return this;
        }

        public Builder withMetricsConf(ServiceMetricConf metricsConf) {
            this.metricConf = metricsConf;
            return this;
        }

        public Builder withValidationMode(ValidationMode validationMode) {
            this.validationMode = validationMode;
            return this;
        }

        public Builder withItemState(ShardState shardState) {
            this.shardState = shardState;
            return this;
        }

        public Builder withMetricNameLabel(String metricNameLabel) {
            this.metricNameLabel = metricNameLabel;
            return this;
        }

        public CoremonShardOpts build() {
            return new CoremonShardOpts(this);
        }
    }
}
