package ru.yandex.solomon.coremon;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;

import ru.yandex.solomon.core.conf.ShardConfDetailed;

/**
 * @author Sergey Polovko
 */
interface CoremonStateEvent {

    /**
     * Finishes event processing.
     */
    void done();

    /**
     * Aborts event processing.
     */
    void abort(String message);

    /**
     * Event to reload shard with given config. If {@code awaitLoad == true} , then this event must be completed
     * twice before extended future will be completed: 1) after actual shard load is done and 2) after reference to
     * a shards map gets updated.
     *
     * Extends CompletableFuture to reduce allocations.
     */
    class ReloadShard extends CompletableFuture<Void> implements CoremonStateEvent {
        private static final AtomicIntegerFieldUpdater<ReloadShard> countDownUpdater =
            AtomicIntegerFieldUpdater.newUpdater(ReloadShard.class, "countDown");

        final ShardConfDetailed conf;
        final boolean awaitLoad;
        final boolean allowUpdate;
        volatile int countDown;

        ReloadShard(ShardConfDetailed conf, boolean allowUpdate, boolean awaitLoad) {
            this.conf = conf;
            this.allowUpdate = allowUpdate;
            this.awaitLoad = awaitLoad;
            this.countDown = awaitLoad ? 2 : 1;
        }

        @Override
        public void done() {
            complete(null);
        }

        @Override
        public boolean complete(Void value) {
            if (countDownUpdater.decrementAndGet(this) <= 0) {
                return super.complete(value);
            }
            return false;
        }

        @Override
        public void abort(String message) {
            super.completeExceptionally(new IllegalStateException(message));
        }
    }

    /**
     * Event to stop and remove shard with given id.
     * Extends CompletableFuture to reduce allocations.
     */
    class RemoveShard extends CompletableFuture<Void> implements CoremonStateEvent {
        final int numId;

        RemoveShard(int numId) {
            this.numId = numId;
        }

        @Override
        public void done() {
            super.complete(null);
        }

        @Override
        public void abort(String message) {
            super.completeExceptionally(new IllegalStateException(message));
        }
    }

    class UpdateConf implements CoremonStateEvent {
        public static final CoremonStateEvent INSTANCE = new UpdateConf();

        @Override
        public void done() {
        }

        @Override
        public void abort(String message) {
        }
    }
}
