package ru.yandex.solomon.coremon;

import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.mutable.MutableInt;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.monitoring.coremon.EShardState;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.histogram.HistogramCollector;
import ru.yandex.monlib.metrics.histogram.HistogramSnapshot;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.config.protobuf.coremon.TCoremonMetabaseConfig;
import ru.yandex.solomon.coremon.meta.service.MetabaseNotInitialized;
import ru.yandex.solomon.coremon.stockpile.CoremonShardStockpileMetrics;
import ru.yandex.solomon.selfmon.counters.EnumMetrics;


/**
 * @author Sergey Polovko
 */
@Component
public class CoremonStateMetrics implements MetricSupplier {

    private final CoremonState state;
    private final boolean verboseMetrics;

    @Autowired
    public CoremonStateMetrics(CoremonState state, TCoremonMetabaseConfig config) {
        this.state = state;
        this.verboseMetrics = config.getReportShardVerboseMetrics();
    }

    @Override
    public int estimateCount() {
        return state.getShardsCount() * CoremonShardStockpileMetrics.COUNT;
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        try {
            CoremonShardStockpileMetrics totalMetrics = new CoremonShardStockpileMetrics("total", "total");
            Map<String, CoremonShardStockpileMetrics> metricsByProjectId = new HashMap<>(10);

            var statesCount = EnumMetrics.gaugesInt64(
                EShardState.class,
                totalMetrics.getRegistry(),
                "engine.shard.count",
                "state");

            HistogramCollector shardSizes = Histograms.exponential(30, 2);

            MutableInt nonReadyShards = new MutableInt(0);
            state.getInitializedShards().forEach(s -> {
                shardSizes.collect(s.getMetabaseShard().getStorage().getRowsLoaded());

                if (!s.isReady()) {
                    nonReadyShards.increment();
                }

                String projectId = s.getProcessingShard().getShardKey().getProject();
                statesCount.get(s.getMetabaseShard().getState()).add(1);

                CoremonShardStockpileMetrics metrics = s.getProcessingShard().updateAndGetMetrics(s.getMetabaseShard());
                totalMetrics.combine(metrics);

                metrics.append(tsMillis, commonLabels, consumer, verboseMetrics);

                CoremonShardStockpileMetrics projectMetrics =
                    metricsByProjectId.computeIfAbsent(
                        projectId,
                        _projectId -> new CoremonShardStockpileMetrics(_projectId, "total")
                    );

                projectMetrics.combine(metrics);
            });

            for (CoremonShardStockpileMetrics projectMetrics : metricsByProjectId.values()) {
                projectMetrics.append(tsMillis, commonLabels, consumer, verboseMetrics);
            }

            totalMetrics.append(tsMillis, commonLabels, consumer, true); // always report all aggregated metrics

            appendGauge(tsMillis, commonLabels, consumer, "engine.nonReadyShards", nonReadyShards.intValue());
            appendHist(tsMillis, commonLabels, consumer, "engine.shard.sizes", shardSizes.snapshot());
        } catch (MetabaseNotInitialized ignored) {
            // avoid noise in logs
        }
    }

    private void appendHist(long tsMillis, Labels commonLabels, MetricConsumer c, String name, HistogramSnapshot hist) {
        c.onMetricBegin(MetricType.HIST);
        {
            c.onLabelsBegin(commonLabels.size() + 1);
            {
                commonLabels.forEach(c::onLabel);
                c.onLabel("sensor", name);
            }
            c.onLabelsEnd();
            c.onHistogram(tsMillis, hist);
        }
        c.onMetricEnd();
    }

    private void appendGauge(long tsMillis, Labels commonLabels, MetricConsumer c, String name, double value) {
        c.onMetricBegin(MetricType.DGAUGE);
        {
            c.onLabelsBegin(commonLabels.size() + 1);
            {
                commonLabels.forEach(c::onLabel);
                c.onLabel("sensor", name);
            }
            c.onLabelsEnd();
            c.onDouble(tsMillis, value);
        }
        c.onMetricEnd();
    }
}
