package ru.yandex.solomon.coremon;

import java.time.Instant;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;

import ru.yandex.monitoring.coremon.EShardState;
import ru.yandex.solomon.coremon.stockpile.CoremonShardStockpile;
import ru.yandex.solomon.staffOnly.html.AHref;
import ru.yandex.solomon.staffOnly.html.HtmlError;
import ru.yandex.solomon.staffOnly.html.HtmlProgressBar;
import ru.yandex.solomon.staffOnly.html.HtmlWriter;
import ru.yandex.solomon.staffOnly.html.HtmlWriterWithCommonLibraries;
import ru.yandex.solomon.staffOnly.manager.ExtraContentParam;
import ru.yandex.solomon.staffOnly.manager.ManagerController;
import ru.yandex.solomon.staffOnly.manager.WithAttrs;
import ru.yandex.solomon.util.ExceptionUtils;
import ru.yandex.solomon.util.async.InFlightLimiter;


/**
 * @author Sergey Polovko
 */
@SuppressWarnings("Duplicates")
final class CoremonStateWww {
    private CoremonStateWww() {}

    static void shardsPage(
        ExtraContentParam p,
        InFlightLimiter inFlightLimiter,
        Collection<CoremonShard> shards,
        Int2ObjectMap<Throwable> errors)
    {
        HtmlWriterWithCommonLibraries hw = p.getHtmlWriter();

        p.subh("In-flight limiter");
        p.managerWriter().getHtmlWriter().p(String.format(
            "[current: %d, waiting: %d]",
            inFlightLimiter.getCurrent(),
            inFlightLimiter.getWaitingCount()));

        {
            List<LoadingRow> rows = shards.stream()
                .filter(s -> !s.isReady())
                .sorted(Comparator.comparing(CoremonShard::getId))
                .map(LoadingRow::new)
                .collect(Collectors.toList());

            if (!rows.isEmpty()) {
                p.subh(String.format("Loading Shards (%d)", rows.size()));
                p.managerWriter().reflectListTable(rows, LoadingRow.class);
            }
        }

        {
            List<LoadedRow> rows = shards.stream()
                .filter(CoremonShard::isReady)
                .sorted(Comparator.comparing(CoremonShard::getId))
                .map(LoadedRow::fromShard)
                .collect(Collectors.toList());

            p.subh("Loaded Shards");
            if (shards.isEmpty()) {
                hw.p("no shards");
            } else {
                rows.add(0, LoadedRow.sum(rows));
                p.managerWriter().reflectListTable(rows, LoadedRow.class);
            }
        }

        p.subh("Failed Shards");
        if (errors.isEmpty()) {
            hw.p("no shards");
        } else {
            hw.tableTable(() -> {
                hw.trThs("Id", "Exception");

                for (var e : errors.int2ObjectEntrySet()) {
                    int numId = e.getIntKey();
                    Throwable error = e.getValue();
                    hw.tr(() -> {
                        hw.tdText(Integer.toUnsignedString(numId));
                        hw.td(() -> hw.preText(ExceptionUtils.printStackTrace(error)));
                    });
                }
            });
        }
    }

    /**
     * LOADING ROW
     */
    private static final class LoadingRow {
        private final Object numId;
        private final Object shardId;
        private final long loadedRows;
        private final long totalRows;
        private final Object progress;
        private final EShardState state;

        LoadingRow(CoremonShard s) {
            this.numId = new AHref(ManagerController.namedObjectLink(s.namedObjectIdGlobal()), Integer.toUnsignedString(s.getNumId()));
            this.shardId = new AHref(ManagerController.namedObjectLink(s.namedObjectIdGlobal()), s.getId());
            this.loadedRows = s.getMetabaseShard().getStorage().getRowsLoaded();
            this.totalRows = s.getMetabaseShard().getStorage().getEstimatedRowsTotal();
            this.progress = new HtmlProgressBar((loadedRows * 100.0) / (double) totalRows);
            this.state = s.getProcessingShard().getState();
        }
    }

    /**
     * LOADED ROW
     */
    private static final class LoadedRow implements WithAttrs {

        private final Object numId;
        private final Object shardId;
        private final int memAndDiskMetrics;
        private final long metricsTotal;
        private final int inFlight;
        private final Object lastError;

        LoadedRow(Object numId, Object shardId, int memAndDiskMetrics, long metricsTotal, int inFlight, Object lastError) {
            this.numId = numId;
            this.shardId = shardId;
            this.memAndDiskMetrics = memAndDiskMetrics;
            this.metricsTotal = metricsTotal;
            this.inFlight = inFlight;
            this.lastError = lastError;
        }

        static LoadedRow fromShard(CoremonShard s) {
            Object numId = new AHref(ManagerController.namedObjectLink(s.namedObjectIdGlobal()), Integer.toUnsignedString(s.getNumId()));
            Object shardId = new AHref(ManagerController.namedObjectLink(s.namedObjectIdGlobal()), s.getId());
            CoremonShardStockpile shard = s.getProcessingShard();
            int memAndDiskMetrics = s.getMetabaseShard().fileAndMemOnlyMetrics();
            long metricsTotal = shard.getMetrics().getMetricsTotal();
            int inFlight = s.inFlight();

            Object lastError = (shard.getLastException() != null)
                ? new HtmlError(Instant.ofEpochMilli(shard.getLastExceptionInstant()), shard.getLastException())
                : null;

            return new LoadedRow(numId, shardId, memAndDiskMetrics, metricsTotal, inFlight, lastError);
        }

        static LoadedRow sum(List<LoadedRow> rows) {
            Object numId = "total";
            Object shardId = "total";
            int memAndDiskMetrics = rows.stream().mapToInt(r -> r.memAndDiskMetrics).sum();
            long metricsTotal = rows.stream().mapToLong(r -> r.metricsTotal).sum();
            int inFlight = rows.stream().mapToInt(r -> r.inFlight).sum();
            return new LoadedRow(numId, shardId, memAndDiskMetrics, metricsTotal, inFlight, null);
        }

        @Override
        public HtmlWriter.Attr[] attrs() {
            if (lastError != null) {
                return new HtmlWriter.Attr[]{
                    HtmlWriter.Attr.cssClass("warning")
                };
            }
            return HtmlWriter.EMPTY_ATTRS;
        }
    }
}
