package ru.yandex.solomon.coremon.aggregates;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.core.conf.aggr.AggrRuleConf;
import ru.yandex.solomon.core.db.model.MetricAggregation;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.stockpile.CoremonShardStockpileResolveHelper;
import ru.yandex.solomon.coremon.stockpile.CoremonShardStockpileWriteHelper;
import ru.yandex.solomon.coremon.stockpile.write.UnresolvedMetricPoint;
import ru.yandex.solomon.coremon.stockpile.write.UnresolvedMetricTimeSeries;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.CountColumn;
import ru.yandex.solomon.model.point.column.MergeColumn;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.api.EDecimPolicy;


/**
 * @author Sergey Polovko
 */
public class AggrHelper {
    private static final ru.yandex.solomon.model.protobuf.MetricType WRITE_TYPE = ru.yandex.solomon.model.protobuf.MetricType.DGAUGE;

    private final AggrRuleConf[] aggrRules;
    private final EDecimPolicy decimPolicyId;
    private final LabelAllocator labelAllocator;

    public AggrHelper(AggrRuleConf[] aggrRules, EDecimPolicy decimPolicyId, LabelAllocator labelAllocator) {
        this.aggrRules = aggrRules;
        this.decimPolicyId = decimPolicyId;
        this.labelAllocator = labelAllocator;
    }

    public AggrHelper(AggrRuleConf[] aggrRules, EDecimPolicy decimPolicyId) {
        this(aggrRules, decimPolicyId, Labels.allocator);
    }

    public boolean hasAggrRules() {
        return aggrRules.length > 0;
    }

    /**
     * @return {@code true} if aggregates was updated, {@code false} otherwise
     */
    public boolean aggregatePoint(
        CoremonShardStockpileResolveHelper resolveHelper,
        CoremonShardStockpileWriteHelper writeHelper,
        Labels labels,
        AggrPoint point,
        MetricType type,
        ru.yandex.solomon.model.protobuf.MetricType writeType, // TODO: drop writeType as only all coremon shards will be write type to stockpile as for metabase (gordiychuk@)
        CoremonMetric coremonMetric,
        Labels optLabels)
    {
        Object aggrMetrics = coremonMetric.getAggrMetrics();

        if (Double.isNaN(point.getValueNum()) || AggrMetrics.isEmpty(aggrMetrics)) {
            return false;
        }

        final int optLabelsHash = optLabels.hashCode();
        if (AggrMetrics.isInitialized(aggrMetrics)) {
            //
            // Check that opt labels are not changed.
            //
            // There are cases when host can be moved from one DC to another without
            // changing FQDN. And if user uses aggregation rule like this [host=*] => [host={{DC}}]
            // (with opt labels substitution) resolved aggregate metrics IDs will not
            // be updated after changes in opt labels.
            //
            int[] packedAggrs = (int[]) aggrMetrics;
            if (optLabelsHash == AggrMetrics.optLabelsHash(packedAggrs)) {
                writeResolvedPoint(packedAggrs, point, writeType, writeHelper);
                return false;
            }
        }

        Map<Labels, MetricAggregation> targets = targetMetricsSet(labels, optLabels);
        if (targets.isEmpty()) {
            // no aggregates after trying to apply aggregation rules
            coremonMetric.setAggrMetrics(AggrMetrics.EMPTY_AGGRS);
            return true;
        }

        Object maybeResolved = resolvedAndWritePoint(
            optLabelsHash,
            targets, resolveHelper, writeHelper, type, writeType, point);

        if (AggrMetrics.isInitialized(maybeResolved)) {
            coremonMetric.setAggrMetrics(maybeResolved);
            return true;
        }

        return false;
    }

    /**
     * @return {@code true} if aggregates was updated, {@code false} otherwise
     */
    public boolean aggregateTimeseries(
            CoremonShardStockpileResolveHelper resolveHelper,
            CoremonShardStockpileWriteHelper writeHelper,
            Labels labels,
            MetricType type,
            AggrPoint tempPoint,
            AggrGraphDataArrayList timeSeries,
            long gridMillis,
            CoremonMetric coremonMetric,
            Labels optLabels)
    {
        Object aggrMetrics = coremonMetric.getAggrMetrics();
        if (!hasAggrRules() || AggrMetrics.isEmpty(aggrMetrics)) {
            return false;
        }

        final int optLabelsHash = optLabels.hashCode();
        if (AggrMetrics.isInitialized(aggrMetrics)) {
            //
            // Check that opt labels are not changed.
            //
            // There are cases when host can be moved from one DC to another without
            // changing FQDN. And if user uses aggregation rule like this [host=*] => [host={{DC}}]
            // (with opt labels substitution) resolved aggregate metrics IDs will not
            // be updated after changes in opt labels.
            //
            int[] packedAggrs = (int[]) aggrMetrics;
            if (optLabelsHash == AggrMetrics.optLabelsHash(packedAggrs)) {
                writeResolvedTimeseries(packedAggrs, tempPoint, timeSeries, gridMillis, writeHelper);
                return false;
            }
        }

        Map<Labels, MetricAggregation> targets = targetMetricsSet(labels, optLabels);
        if (targets.isEmpty()) {
            // no aggregates after trying to apply aggregation rules
            coremonMetric.setAggrMetrics(AggrMetrics.EMPTY_AGGRS);
            return true;
        }

        Object maybeResolved = resolveAndWriteTimeseries(
                optLabelsHash,
                targets, resolveHelper, writeHelper, type, tempPoint, timeSeries, gridMillis);

        if (AggrMetrics.isInitialized(maybeResolved)) {
            coremonMetric.setAggrMetrics(maybeResolved);
            return true;
        }

        Object aggrMetricsNew = coremonMetric.getAggrMetrics();
        return AggrMetrics.isInitialized(aggrMetricsNew) && aggrMetrics != aggrMetricsNew;
    }

    private void writeResolvedPoint(
        int[] aggrMetrics,
        AggrPoint point,
        ru.yandex.solomon.model.protobuf.MetricType writeType,
        CoremonShardStockpileWriteHelper writer)
    {
        for (int i = 0, size = AggrMetrics.size(aggrMetrics); i < size; i++) {
            writeResolvedPoint(
                    writer,
                    AggrMetrics.aggr(aggrMetrics, i),
                    point,
                    AggrMetrics.shardId(aggrMetrics, i),
                    AggrMetrics.localId(aggrMetrics, i),
                    writeType);

            // aggrMetrics is a copy of the data stored in FileMetricsCollection
            // and updating it's last point time will be wasted
        }
    }

    private void writeResolvedTimeseries(
            int[] aggrMetrics,
            AggrPoint point,
            AggrGraphDataArrayList timeSeries,
            long gridMillis,
            CoremonShardStockpileWriteHelper writeHelper)
    {
        MetricAggregation lastAggr = null;
        int timeseriesSize = timeSeries.length();

        for (int metricIdx = 0, size = AggrMetrics.size(aggrMetrics); metricIdx < size; metricIdx++) {
            var aggr = AggrMetrics.aggr(aggrMetrics, metricIdx);
            if (lastAggr != aggr) {
                lastAggr = aggr;
                timeSeries.truncate(timeseriesSize);
                MetricAggregator.aggregate(point, timeSeries, timeseriesSize, gridMillis, aggr);
            }

            int shardId = AggrMetrics.shardId(aggrMetrics, metricIdx);
            long localId = AggrMetrics.localId(aggrMetrics, metricIdx);
            writeResolvedTimeseries(writeHelper, aggr, point, timeSeries, timeseriesSize, shardId, localId);

            // aggrMetrics is a copy of the data stored in FileMetricsCollection
            // and updating it's last point time will be wasted
        }

        timeSeries.truncate(timeseriesSize);
    }

    @Nullable
    private Object resolvedAndWritePoint(
        int optLabelsHash,
        Map<Labels, MetricAggregation> targets,
        CoremonShardStockpileResolveHelper resolveHelper,
        CoremonShardStockpileWriteHelper writeHelper,
        MetricType type,
        ru.yandex.solomon.model.protobuf.MetricType writeType,
        AggrPoint point)
    {
        var resolved = new ResolvedMetrics(optLabelsHash, targets.size());
        var it = targets.entrySet()
                .stream()
                .sorted(Entry.comparingByValue())
                .iterator();

        while (it.hasNext()) {
            var entry = it.next();
            var labels = entry.getKey();
            var aggr = entry.getValue();
            try (CoremonMetric targetMetric = resolveHelper.tryResolveMetric(labels, type)) {
                if (targetMetric != null) {
                    int shardId = targetMetric.getShardId();
                    long localId = targetMetric.getLocalId();
                    writeResolvedPoint(writeHelper, aggr, point, shardId, localId, writeType);
                    targetMetric.setLastPointSeconds(InstantUtils.millisecondsToSeconds(point.tsMillis));

                    resolved.add(shardId, localId, aggr);
                } else if (!resolveHelper.reachQuota()) {
                    writeUnresolvedPoint(resolveHelper, aggr, point, labels, type, writeType);
                }
            }
        }

        return resolved.pack();
    }

    @Nullable
    private Object resolveAndWriteTimeseries(
            int optLabelsHash,
            Map<Labels, MetricAggregation> targets,
            CoremonShardStockpileResolveHelper resolveHelper,
            CoremonShardStockpileWriteHelper writeHelper,
            MetricType type,
            AggrPoint point,
            AggrGraphDataArrayList timeSeries,
            long gridMillis)
    {
        var resolved = new ResolvedMetrics(optLabelsHash, targets.size());

        MetricAggregation lastAggr = null;
        int timeseriesSize = timeSeries.length();

        var it = targets.entrySet()
                .stream()
                .sorted(Entry.comparingByValue())
                .iterator();

        while (it.hasNext()) {
            var entry = it.next();
            var labels = entry.getKey();
            var aggr = entry.getValue();

            if (lastAggr != aggr) {
                lastAggr = aggr;
                timeSeries.truncate(timeseriesSize);
                MetricAggregator.aggregate(point, timeSeries, timeseriesSize, gridMillis, aggr);
            }

            try (CoremonMetric targetMetric = resolveHelper.tryResolveMetric(labels, type)) {
                if (targetMetric != null) {
                    int shardId = targetMetric.getShardId();
                    long localId = targetMetric.getLocalId();

                    writeResolvedTimeseries(writeHelper, aggr, point, timeSeries, timeseriesSize, shardId, localId);
                    targetMetric.setLastPointSeconds(InstantUtils.millisecondsToSeconds(point.tsMillis));

                    resolved.add(shardId, localId, aggr);
                } else if (!resolveHelper.reachQuota()) {
                    writeUnresolvedTimeseries(resolveHelper, aggr, point, timeSeries, timeseriesSize, labels, type);
                }
            }
        }

        timeSeries.truncate(timeseriesSize);
        return resolved.pack();
    }

    @Nonnull
    public Map<Labels, MetricAggregation> targetMetricsSet(Labels metricLabels, Labels optLabels) {
        if (aggrRules.length == 0) {
            return Map.of();
        }

        var targets = new HashMap<Labels, MetricAggregation>(aggrRules.length);
        for (AggrRuleConf aggrRule : aggrRules) {
            Labels target = aggrRule.matchTransform(metricLabels, optLabels, labelAllocator);
            if (target != null) {
                targets.put(target, aggrRule.getAggregation());
            }
        }

        return targets;
    }

    private void fillAggregation(AggrPoint point, MetricAggregation aggregation) {
        point.setMerge(aggregation == MetricAggregation.SUM);
        point.setCount(1);
    }

    private void writeResolvedTimeseries(CoremonShardStockpileWriteHelper writer, MetricAggregation aggr, AggrPoint point, AggrGraphDataArrayList timeSeries, int from, int shardId, long localId) {
        for (int pointIdx = from; pointIdx < timeSeries.length(); pointIdx++) {
            timeSeries.getDataTo(pointIdx, point);
            fillAggregation(point, aggr);
            writer.addPoint(shardId, localId, point, decimPolicyId.getNumber(), WRITE_TYPE);
        }
    }

    private void writeUnresolvedTimeseries(CoremonShardStockpileResolveHelper writer, MetricAggregation aggr, AggrPoint point, AggrGraphDataArrayList timeSeries, int from, Labels labels, MetricType type) {
        int mask = timeSeries.columnSetMask() | MergeColumn.mask | CountColumn.mask;
        var timeseriesCopy = new AggrGraphDataArrayList(mask, timeSeries.length() - from);
        for (int pointIdx = from; pointIdx < timeSeries.length(); pointIdx++) {
            timeSeries.getDataTo(pointIdx, point);
            fillAggregation(point, aggr);
            timeseriesCopy.addRecord(point);
        }

        Label hostLabel = labels.findByKey(LabelKeys.HOST);
        String targetHost = (hostLabel == null) ? "" : hostLabel.getValue();

        var unresolvedPoint = new UnresolvedMetricTimeSeries(labels, type, WRITE_TYPE, timeseriesCopy);
        writer.addAggrData(targetHost, unresolvedPoint);
    }

    private void writeResolvedPoint(CoremonShardStockpileWriteHelper writer, MetricAggregation aggr, AggrPoint point, int shardId, long localId, ru.yandex.solomon.model.protobuf.MetricType writeType) {
        fillAggregation(point, aggr);
        writer.addPoint(shardId, localId, point, decimPolicyId.getNumber(), writeType);
    }

    private void writeUnresolvedPoint(CoremonShardStockpileResolveHelper writer, MetricAggregation aggr, AggrPoint point, Labels labels, MetricType type, ru.yandex.solomon.model.protobuf.MetricType writeType) {
        Label hostLabel = labels.findByKey(LabelKeys.HOST);
        String targetHost = (hostLabel == null) ? "" : hostLabel.getValue();

        fillAggregation(point, aggr);
        UnresolvedMetricPoint metricPoint = new UnresolvedMetricPoint(labels, type, writeType, point);
        writer.addAggrData(targetHost, metricPoint);
    }

    private static class ResolvedMetrics {
        private final int optLabelsHash;
        private final int[] shardIds;
        private final long[] localIds;
        private final MetricAggregation[] aggregations;
        private int size;

        public ResolvedMetrics(int optLabelsHash, int capacity) {
            this.optLabelsHash = optLabelsHash;
            this.shardIds = new int[capacity];
            this.localIds = new long[capacity];
            this.aggregations = new MetricAggregation[capacity];
            this.size = 0;
        }

        public void add(int shardId, long localId, MetricAggregation aggregation) {
            shardIds[size] = shardId;
            localIds[size] = localId;
            aggregations[size] = aggregation;
            size++;
        }

        public int[] pack() {
            if (size < shardIds.length) {
                return null;
            }

            return AggrMetrics.pack(optLabelsHash, aggregations, shardIds, localIds);
        }
    }
}
