package ru.yandex.solomon.coremon.aggregates;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.lang.Verify;
import ru.yandex.solomon.core.db.model.MetricAggregation;
import ru.yandex.stockpile.client.shard.StockpileLocalId;
import ru.yandex.stockpile.client.shard.StockpileShardId;


/**
 * To reduce allocations we store resolved aggregates data in single int array using next format:
 *
 *    32 bits      32 bits      32 bits      32 bits        32 bits                  32 bits      32 bits      32 bits      32 bits
 * +------------+------------+------------+------------+------------+------------+------------+------------+------------+------------+
 * | opt labels |  aggr_1    |  shardId_1 | localId_1  |  localId_1 |            |  aggr_n    | shardId_n  | localId_n  |  localId_b |
 * | hash code  |            |            | (lo part)  |  (hi part) |     ...    |            |            | (lo part)  |  (hi part) |
 * +------------+------------+------------+------------+------------+------------+------------+------------+------------+------------+
 *      0             1            2             3           4                       1+4n         1+4n+2       1+4n+3        1+4n+4
 *
 *
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public final class AggrMetrics {
    private static final int BLOCKS = 4;
    private static final int AGGR_ID_IDX = 0;
    private static final int SHARD_ID_IDX = 1;
    private static final int LOCAL_ID_LO_IDX = 2;

    private AggrMetrics() {}

    /**
     * Special marker for empty aggregates in metric.
     */
    public static final Object EMPTY_AGGRS = new Object();

    /**
     * @return {@code true} iff object representing aggregate metrics is properly initialized yet.
     */
    public static boolean isInitialized(@Nullable Object aggrs) {
        return aggrs instanceof int[];
    }

    /**
     * @return {@code true} iff object representing aggregate metrics is empty.
     */
    public static boolean isEmpty(@Nullable Object aggrs) {
        return aggrs == EMPTY_AGGRS;
    }

    /**
     * Packs given shard and local ids into single array of ints.
     */
    public static int[] pack(int optLabelsHash, MetricAggregation[] aggrs, int[] shardIds, long[] localIds) {
        Verify.isTrue(localIds.length == shardIds.length);
        Verify.isTrue(localIds.length == aggrs.length);
        int[] ids = new int[localIds.length * BLOCKS + 1];
        ids[0] = optLabelsHash;
        for (int i = 0, j = 1; i < shardIds.length; i++) {
            ids[j++] = aggrs[i].ordinal();
            ids[j++] = shardIds[i];
            ids[j++] = (int) (localIds[i] & 0xffffffffL); // lo
            ids[j++] = (int) (localIds[i] >>> 32); // hi
        }
        return ids;
    }

    /**
     * @return number of metrics packed into given array.
     */
    public static int size(int[] ids) {
        return (ids.length - 1) / BLOCKS;
    }

    public static int optLabelsHash(int[] ids) {
        return ids[0];
    }

    public static int shardId(int[] ids, int index) {
        return ids[shiftToBlock(index) + SHARD_ID_IDX];
    }

    public static long localId(int[] ids, int index) {
        final int lo = shiftToBlock(index) + LOCAL_ID_LO_IDX;
        final int hi = lo + 1;
        return (Integer.toUnsignedLong(ids[hi]) << 32) | Integer.toUnsignedLong(ids[lo]);
    }

    public static MetricAggregation aggr(int[] ids, int index) {
        return MetricAggregation.byNumber(ids[shiftToBlock(index) + AGGR_ID_IDX]);
    }

    private static int shiftToBlock(int index) {
        return BLOCKS * index + 1;
    }

    public static String toString(@Nullable Object aggrs) {
        if (isEmpty(aggrs)) {
            return "EMPTY";
        }
        if (isInitialized(aggrs)) {
            int[] ids = (int[]) aggrs;
            final int size = size(ids);
            StringBuilder sb = new StringBuilder(10 * size);
            sb.append('[');
            for (int i = 0; i < size; i++) {
                sb.append(aggr(ids, i));
                sb.append("/");
                sb.append(StockpileShardId.toString(shardId(ids, i)));
                sb.append('/');
                sb.append(StockpileLocalId.toString(localId(ids, i)));
                sb.append(", ");
            }
            if (sb.length() > 2) {
                sb.setLength(sb.length() - 2);
            }
            sb.append(']');
            return sb.toString();
        }

        return "NOT_INITIALIZED";
    }
}
