package ru.yandex.solomon.coremon.aggregates;

import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.ValueColumn;

/**
 * To be able to calculate aggregate over different metrics' values we need stable timestamps,
 * so we align points to some window boundary. After that several points of the same metric
 * (those who hit the window) become to have the same time. So we need somehow to combine
 * their values. Here we use the average function to produce more natural values for a user.
 *
 * But here's the problem. When we receive different timeseries of the metrics which must be
 * aggregated they may not perfectly feet out window and we can get partially filled
 * windows, e.g. (* - means no value, [] - window boundaries):
 *
 *      metric1  [ A, B, * ]
 *      metric2  [ *, C, D,] [ E, F, * ]
 *      metric3              [ *, G, H ], [ I, *, * ]
 *
 * And we need to calculate sum(metric1, metric2, metric3) here:
 *
 *      [ sum(avg(A, B), avg(C, D)), sum(avg(E, F), avg(G, H)), I ]
 *
 * Remember that sum() is implicitly calculated in Stockpile when we write mergeable points
 * with the same timestamp. And if we somehow represent avg() function through the sum()
 * function then all this partially calculated avg() parts will be merged inside Stockpile.
 *
 * If we assign some weight to each point we can calculate avg() as:
 *
 *      avg(Y, Z) = (Y * Wy + Z * Wz) / (Wy + Wz)
 *
 * Where Wx - is the weight of the point X. We can calculate it as difference between neighboring
 * points' timestamps. It can be seen that (Wy + Wz) is approximately equal to a window size,
 * so avg() can be calculated as:
 *
 *      avg(Y, Z) = Y * Wy / WindowSize + Z * Wz / WindowSize
 *
 * And this is exactly what we need here.
 *
 * @author Sergey Polovko
 */
public class AggrWeightAvgCollector {
    private final long windowSizeMillis;

    public AggrWeightAvgCollector(long windowSizeMillis) {
        this.windowSizeMillis = windowSizeMillis;
    }

    private long windowStartMillis;

    private long prevTsMillis;
    private long prevPrevTsMillis;
    private double prevValue;
    private double weightedSum;
    private int count;

    private void reset() {
        windowStartMillis = 0;
        weightedSum = 0.0;
        count = 0;
        prevTsMillis = 0;
        prevPrevTsMillis = 0;
    }

    public boolean append(AggrPoint point) {
        if (windowStartMillis == 0) {
            initWindow(point.tsMillis);
            prevValue = point.getValueDivided();
            prevTsMillis = point.tsMillis;
            return false;
        }

        final double value = prevValue;
        if (!Double.isNaN(value)) {
            long weight = point.tsMillis - prevTsMillis; // next - current

            // Actually there is no reason to divide weight by 1000, because
            // weight calculated in milliseconds and window size (which is
            // written in denominator) also in milliseconds. But...
            // fo reasons unknown for me there is multiplication to 1000 in
            // ValueColumn::divide() method
            weightedSum += value * weight / 1000;
            count++;
        }

        prevPrevTsMillis = this.prevTsMillis;
        prevTsMillis = point.tsMillis;
        prevValue = point.getValueDivided();

        boolean result = false;
        long nextTsMillis = point.tsMillis;
        if (nextTsMillis - windowStartMillis >= windowSizeMillis) {  // went to a window border
            if (count != 0) { // if window contains all NaNs we have to skip it
                point.setTsMillis(windowStartMillis);
                point.setValue(weightedSum, windowSizeMillis);
                point.setMerge(true);
                point.setCount(count);
                result = true;
            }

            initWindow(nextTsMillis);
            weightedSum = 0.0;
            count = 0;
        }

        return result;
    }

    private void initWindow(long tsMillis) {
        windowStartMillis = tsMillis;
        windowStartMillis -= (windowStartMillis % windowSizeMillis); // align window start time
    }

    public boolean compute(AggrPoint point) {
        if (prevTsMillis == 0) {
            return false;
        }

        if (prevPrevTsMillis == 0) {
            // here we can't estimate point weight, so the best that
            // we can do is to write point value as is with aligned
            // timestamp
            point.setTsMillis(windowStartMillis);
            point.setValue(prevValue, ValueColumn.DEFAULT_DENOM);
            point.setMerge(true);
            point.setCount(1);
            return true;
        }

        weightedSum += prevValue * (prevTsMillis - prevPrevTsMillis) / 1000;
        point.setTsMillis(windowStartMillis);
        point.setValue(weightedSum, windowSizeMillis);
        point.setMerge(true);
        point.setCount(count + 1);
        return true;
    }
}
