package ru.yandex.solomon.coremon.aggregates;

import ru.yandex.solomon.core.db.model.MetricAggregation;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricAggregator {

    public static void aggregate(AggrPoint tempPoint, AggrGraphDataArrayList timeSeries, int size, long gridMillis, MetricAggregation aggregation) {
        switch (aggregation) {
            case LAST:
                aggregateLast(tempPoint, timeSeries, size, gridMillis);
                break;
            case SUM:
                aggregationSum(tempPoint, timeSeries, size, gridMillis);
                break;
        }
    }

    public static void aggregate(AggrGraphDataArrayList timeSeries, long gridMillis, MetricAggregation aggregation) {
        var point = RecyclableAggrPoint.newInstance();
        try {
            aggregate(point, timeSeries, timeSeries.length(), gridMillis, aggregation);
        } finally {
            point.recycle();
        }
    }

    public static void aggregateLast(AggrPoint tempPoint, AggrGraphDataArrayList timeSeries, int size, long gridMillis) {
        timeSeries.truncate(size);
        tempPoint.columnSet = timeSeries.columnSetMask();

        double value = 0;
        int count = 0;
        long tsStart = 0;

        for (int index = 0; index < size; index++) {
            long tsMillis = InstantUtils.truncate(timeSeries.getTsMillis(index), gridMillis);
            if (tsStart != tsMillis) {
                if (count != 0) {
                    tempPoint.tsMillis = tsStart;
                    tempPoint.valueNum = value;
                    tempPoint.valueDenom = ValueColumn.DEFAULT_DENOM;
                    timeSeries.addRecord(tempPoint);
                }

                count = 0;
                tsStart = tsMillis;
            }

            value = timeSeries.getValueDivided(index);
            count++;
        }

        if (count != 0) {
            tempPoint.tsMillis = tsStart;
            tempPoint.valueNum = value;
            tempPoint.valueDenom = ValueColumn.DEFAULT_DENOM;
            timeSeries.addRecord(tempPoint);
        }
    }

    public static void aggregationSum(AggrPoint tempPoint, AggrGraphDataArrayList timeseries, int size, long gridMillis) {
        timeseries.truncate(size);
        tempPoint.columnSet = timeseries.columnSetMask();

        WeightedAvgSum.consume(timeseries, size, gridMillis, (tsMillis, num, denom) -> {
            tempPoint.tsMillis = tsMillis;
            tempPoint.valueNum = num;
            tempPoint.valueDenom = denom;
            tempPoint.stepMillis = gridMillis;
            timeseries.addRecord(tempPoint);
        });
    }

}
