package ru.yandex.solomon.coremon.aggregates;

import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.util.time.InstantUtils;


/**
 * @author Sergey Polovko
 */
public final class WeightedAvgSum {
    private WeightedAvgSum() {}

    public interface PointConsumer {
        void accept(long tsMillis, double num, long denom);
    }

    public static void consume(AggrGraphDataArrayList timeSeries, long windowSizeMillis, PointConsumer consumer) {
        consume(timeSeries, timeSeries.length(), windowSizeMillis, consumer);
    }

    /**
     * To be able to calculate aggregate over different metrics' values we need stable timestamps,
     * so we align points to some window boundary. After that several points of the same metric
     * (those who hit the window) become to have the same time. So we need somehow to combine
     * their values. Here we use the average function to produce more natural values for a user.
     *
     * But here's the problem. When we receive different timeseries of the metrics which must be
     * aggregated they may not perfectly feet out window and we can get partially filled
     * windows, e.g. (* - means no value, [] - window boundaries):
     *
     *      metric1  [ A, B, * ]
     *      metric2  [ *, C, D,] [ E, F, * ]
     *      metric3              [ *, G, H ], [ I, *, * ]
     *
     * And we need to calculate sum(metric1, metric2, metric3) here:
     *
     *      [ sum(avg(A, B), avg(C, D)), sum(avg(E, F), avg(G, H)), I ]
     *
     * Remember that sum() is implicitly calculated in Stockpile when we write mergeable points
     * with the same timestamp. And if we somehow represent avg() function through the sum()
     * function then all this partially calculated avg() parts will be merged inside Stockpile.
     *
     * If we assign some weight to each point we can calculate avg() as:
     *
     *      avg(Y, Z) = (Y * Wy + Z * Wz) / (Wy + Wz)
     *
     * Where Wx - is the weight of the point X. We can calculate it as difference between neighboring
     * points' timestamps. It can be seen that (Wy + Wz) is approximately equal to a window size,
     * so avg() can be calculated as:
     *
     *      avg(Y, Z) = Y * Wy / WindowSize + Z * Wz / WindowSize
     *
     * And this is exactly what we need here.
     */
    public static void consume(AggrGraphDataArrayList timeSeries, int size, long windowSizeMillis, PointConsumer consumer) {
        if (size == 1) {
            // here we can't estimate point weight, so the best that
            // we can do is to write point value as is with aligned
            // timestamp
            long tsMillis = InstantUtils.truncate(timeSeries.getTsMillis(0), windowSizeMillis);
            consumer.accept(tsMillis, timeSeries.getValueDivided(0), ValueColumn.DEFAULT_DENOM);
        } else {
            long tsStart = InstantUtils.truncate(timeSeries.getTsMillis(0), windowSizeMillis);

            double weightedSum = 0.0;
            int count = 0;

            for (int i = 0; i < size; i++) {
                final long tsMillis = timeSeries.getTsMillis(i);

                if (tsMillis - tsStart >= windowSizeMillis) {  // went to a window border
                    if (count != 0) { // if window contains all NaNs we have to skip it
                        consumer.accept(tsStart, weightedSum, windowSizeMillis);
                    }

                    tsStart = InstantUtils.truncate(tsMillis, windowSizeMillis);
                    weightedSum = 0.0;
                    count = 0;
                }

                final double value = timeSeries.getValueDivided(i);
                if (!Double.isNaN(value)) {
                    long weight = (i == size - 1)
                        ? tsMillis - timeSeries.getTsMillis(i - 1)  // current - prev
                        : timeSeries.getTsMillis(i + 1) - tsMillis; // next - current

                    if (weight > windowSizeMillis) {
                        weight = windowSizeMillis;
                    }

                    // Actually there is no reason to divide weight by 1000, because
                    // weight calculated in milliseconds and window size (which is
                    // written in denominator) also in milliseconds. But...
                    // fo reasons unknown for me there is multiplication to 1000 in
                    // ValueColumn::divide() method

                    weightedSum += value * weight / 1000;
                    count++;
                }
            }
            if (count != 0) {
                consumer.accept(tsStart, weightedSum, windowSizeMillis);
            }
        }
    }
}
