package ru.yandex.solomon.coremon.api;

import javax.annotation.Nonnull;

import com.google.protobuf.TextFormat;
import io.grpc.Status;
import io.grpc.StatusException;
import io.grpc.StatusRuntimeException;
import io.grpc.stub.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.grpc.utils.InternalGrpcService;
import ru.yandex.monitoring.coremon.CoremonServiceGrpc;
import ru.yandex.monitoring.coremon.DeleteMetricsRequest;
import ru.yandex.monitoring.coremon.DeleteMetricsResponse;
import ru.yandex.monitoring.coremon.TCreateShardRequest;
import ru.yandex.monitoring.coremon.TCreateShardResponse;
import ru.yandex.monitoring.coremon.TDataProcessResponse;
import ru.yandex.monitoring.coremon.TInitShardRequest;
import ru.yandex.monitoring.coremon.TInitShardResponse;
import ru.yandex.monitoring.coremon.TPulledDataRequest;
import ru.yandex.monitoring.coremon.TPushedDataRequest;
import ru.yandex.monitoring.coremon.TReloadShardRequest;
import ru.yandex.monitoring.coremon.TReloadShardResponse;
import ru.yandex.monitoring.coremon.TRemoveShardRequest;
import ru.yandex.monitoring.coremon.TRemoveShardResponse;
import ru.yandex.monitoring.coremon.TShardAssignmentsRequest;
import ru.yandex.monitoring.coremon.TShardAssignmentsResponse;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.urlStatus.UrlStatusTypes;


/**
 * @author Sergey Polovko
 */
@Component
@Import(CoremonServiceImpl.class)
public class CoremonGrpcService extends CoremonServiceGrpc.CoremonServiceImplBase implements InternalGrpcService {
    private static final Logger logger = LoggerFactory.getLogger(CoremonGrpcService.class);

    private final CoremonServiceImpl impl;

    @Autowired
    public CoremonGrpcService(CoremonServiceImpl impl) {
        this.impl = impl;
    }

    @Override
    public void processPulledData(TPulledDataRequest request, StreamObserver<TDataProcessResponse> observer) {
        int numId = request.getNumId();
        impl.processPulledData(request)
            .whenComplete((r, t) -> {
                if (t != null) {
                    logger.warn("cannot process pulled data in shard {}", Integer.toUnsignedString(numId), t);
                    r = errorResponse(t);
                }
                observer.onNext(r);
                observer.onCompleted();
            });
    }

    @Override
    public void processPushedData(TPushedDataRequest request, StreamObserver<TDataProcessResponse> observer) {
        int numId = request.getNumId();
        impl.processPushedData(request)
            .whenComplete((r, t) -> {
                if (t != null) {
                    logger.warn("cannot process pushed data in shard {}", Integer.toUnsignedString(numId), t);
                    r = errorResponse(t);
                }
                observer.onNext(r);
                observer.onCompleted();
            });
    }

    @Override
    public void initShard(TInitShardRequest request, StreamObserver<TInitShardResponse> observer) {
        impl.initShard(request)
            .whenComplete((response, t) -> {
                if (t == null) {
                    observer.onNext(response);
                    observer.onCompleted();
                } else {
                    logger.error("cannot init shard {}", TextFormat.shortDebugString(request), t);
                    observer.onError(classifyError(t).asRuntimeException());
                }
            });
    }

    @Override
    public void createShard(TCreateShardRequest request, StreamObserver<TCreateShardResponse> observer) {
        impl.createShard(request)
            .whenComplete((response, t) -> {
                if (t == null) {
                    observer.onNext(response);
                    observer.onCompleted();
                } else {
                    logger.error("cannot create shard {}", TextFormat.shortDebugString(request), t);
                    observer.onError(classifyError(t).asRuntimeException());
                }
            });
    }

    @Override
    public void removeShard(TRemoveShardRequest request, StreamObserver<TRemoveShardResponse> observer) {
        impl.removeShard(request.getProjectId(), request.getShardId(), request.getNumId())
            .whenComplete((taskId, t) -> {
                if (t == null) {
                    observer.onNext(TRemoveShardResponse.newBuilder().setTaskId(taskId).build());
                    observer.onCompleted();
                } else {
                    logger.error("cannot remove shard {}", TextFormat.shortDebugString(request), t);
                    observer.onError(classifyError(t).asRuntimeException());
                }
            });
    }

    @Override
    public void reloadShard(TReloadShardRequest request, StreamObserver<TReloadShardResponse> observer) {
        impl.reloadShard(request.getProjectId(), request.getShardId(), request.getAllowUpdate(), request.getAwaitLoad())
            .whenComplete((aVoid, t) -> {
                if (t == null) {
                    observer.onNext(TReloadShardResponse.getDefaultInstance());
                    observer.onCompleted();
                } else {
                    logger.error("cannot reload shard {}", TextFormat.shortDebugString(request), t);
                    observer.onError(classifyError(t).asRuntimeException());
                }
            });
    }

    @Override
    public void getShardAssignments(TShardAssignmentsRequest request, StreamObserver<TShardAssignmentsResponse> observer) {
        impl.getShardAssignments(request)
            .whenComplete((response, t) -> {
                if (t == null) {
                    observer.onNext(response);
                    observer.onCompleted();
                } else {
                    logger.error("cannot get shard assignments {}", TextFormat.shortDebugString(request), t);
                    observer.onError(classifyError(t).asRuntimeException());
                }
            });
    }

    @Override
    public void deleteMetrics(DeleteMetricsRequest request, StreamObserver<DeleteMetricsResponse> observer) {
        impl.deleteMetrics(request)
            .whenComplete((taskId, t) -> {
                if (t == null) {
                    observer.onNext(DeleteMetricsResponse.newBuilder().setTaskId(taskId).build());
                    observer.onCompleted();
                } else {
                    logger.error("cannot delete metrics {}", TextFormat.shortDebugString(request), t);
                    observer.onError(classifyError(t).asRuntimeException());
                }
            });
    }

    private static Status classifyError(@Nonnull Throwable e) {
        var cause = e;
        while (cause != null) {
            if (cause instanceof StatusException) {
                return ((StatusException) cause).getStatus();
            } else if (cause instanceof StatusRuntimeException) {
                return ((StatusRuntimeException) cause).getStatus();
            } else if (cause instanceof BadRequestException) {
                return Status.INVALID_ARGUMENT.withDescription(cause.getMessage()).withCause(e);
            }

            cause = cause.getCause();
        }

        return Status.INTERNAL.withCause(e);
    }

    private static TDataProcessResponse errorResponse(Throwable t) {
        return TDataProcessResponse.newBuilder()
            .setStatus(UrlStatusTypes.classifyError(t))
            .setErrorMessage(t.getMessage())
            .build();
    }
}
