package ru.yandex.solomon.coremon.api;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import io.grpc.Status;
import io.netty.buffer.Unpooled;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.monitoring.coremon.DeleteMetricsRequest;
import ru.yandex.monitoring.coremon.TCreateShardRequest;
import ru.yandex.monitoring.coremon.TCreateShardResponse;
import ru.yandex.monitoring.coremon.TDataProcessResponse;
import ru.yandex.monitoring.coremon.TInitShardRequest;
import ru.yandex.monitoring.coremon.TInitShardResponse;
import ru.yandex.monitoring.coremon.TPulledDataRequest;
import ru.yandex.monitoring.coremon.TPushedDataRequest;
import ru.yandex.monitoring.coremon.TReloadShardRequest;
import ru.yandex.monitoring.coremon.TShardAssignmentsRequest;
import ru.yandex.monitoring.coremon.TShardAssignmentsResponse;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.core.conf.UnknownShardException;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.coremon.CoremonState;
import ru.yandex.solomon.coremon.balancer.ShardBalancer;
import ru.yandex.solomon.coremon.balancer.cluster.CoremonHost;
import ru.yandex.solomon.coremon.client.ShardAssignmentsSerializer;
import ru.yandex.solomon.coremon.client.grpc.CoremonHostClient;
import ru.yandex.solomon.coremon.shards.ShardCreator;
import ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsCheckTaskHandler;
import ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsMoveTaskHandler;
import ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsRollbackTaskHandler;
import ru.yandex.solomon.coremon.tasks.deleteMetrics.DeleteMetricsTerminateTaskHandler;
import ru.yandex.solomon.coremon.tasks.removeShard.RemoveShardTaskHandler;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.scheduler.ProgressOperator;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.scheduler.TaskScheduler;
import ru.yandex.solomon.util.host.HostUtils;
import ru.yandex.solomon.util.protobuf.ByteStrings;
import ru.yandex.solomon.util.time.InstantUtils;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;


/**
 * @author Stepan Koltsov
 */
@Component
@Import({ CoremonPeers.class })
class CoremonServiceImpl implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(CoremonServiceImpl.class);

    private final CoremonState coremonState;
    private final ShardBalancer balancer;
    private final ShardCreator creator;
    private final CoremonPeers peers;
    private final TaskScheduler scheduler;

    @Autowired
    public CoremonServiceImpl(
        CoremonState coremonState,
        ShardBalancer balancer,
        ShardCreator creator,
        CoremonPeers peers,
        TaskScheduler scheduler)
    {
        this.coremonState = coremonState;
        this.balancer = balancer;
        this.creator = creator;
        this.peers = peers;
        this.scheduler = scheduler;
    }

    CompletableFuture<TDataProcessResponse> processPulledData(TPulledDataRequest request) {
        try {
            var shard = coremonState.getShardByNumIdOrNull(request.getNumId());
            if (shard == null) {
                return unknownShard();
            }

            var optLabels = LabelConverter.protoToLabels(request.getOptionalLabels().getLabelsList(), shard.labelAllocator());
            return shard.push(
                    request.getHost(),
                    optLabels,
                    request.getFormat(),
                    request.getResponseTimeMillis(),
                    ByteStrings.toByteBuf(request.getResponse()),
                    request.getPrevResponseTimeMillis(),
                    ByteStrings.toByteBuf(request.getPrevResponse()),
                    false, false);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    CompletableFuture<TDataProcessResponse> processPushedData(TPushedDataRequest request) {
        try {
            var shard = coremonState.getShardByNumIdOrNull(request.getNumId());
            if (shard == null) {
                return unknownShard();
            }

            long timeMillis = request.getTimeMillis();
            if (timeMillis == 0) {
                timeMillis = InstantUtils.secondsToMillis(InstantUtils.currentTimeSeconds());
            }

            return shard.push(
                    "",
                    Labels.empty(),
                    request.getFormat(),
                    timeMillis,
                    ByteStrings.toByteBuf(request.getContent()),
                    0,
                    Unpooled.EMPTY_BUFFER,
                    true,
                    request.getOnlyNewFormatWrites());
        }  catch (Throwable t) {
            return failedFuture(t);
        }
    }

    CompletableFuture<String> removeShard(String projectId, String shardId, int numId) {
        try {
            logger.info("removing shard {}/{} ({})", projectId, shardId, Integer.toUnsignedString(numId));
            var removeTask = RemoveShardTaskHandler.removeShardTask(projectId, shardId, numId);
            var taskId = removeTask.id();
            return scheduler.schedule(removeTask)
                    .thenCompose(ignore -> {
                        var shard = coremonState.getShardByNumIdOrNull(numId);
                        if (shard == null) {
                            return completedFuture(null);
                        }

                        // TODO: notify deletion manager
                        return coremonState.removeShard(numId);
                    })
                    .thenApply(ignore -> taskId);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    CompletableFuture<Void> reloadShard(String projectId, String shardId, boolean allowUpdate, boolean awaitLoad) {
        return coremonState.reloadShard(projectId, shardId, allowUpdate, awaitLoad);
    }

    private static CompletableFuture<TDataProcessResponse> unknownShard() {
        return completedFuture(TDataProcessResponse.newBuilder()
                .setStatus(UrlStatusType.UNKNOWN_SHARD)
                .setErrorMessage(UnknownShardException.IN_NOT_INITIALIZED_MESSAGE)
                .build());
    }

    CompletableFuture<TShardAssignmentsResponse> getShardAssignments(TShardAssignmentsRequest request) {
        try {
            Optional<String> leaderHost = balancer.getLeaderHost();
            if (leaderHost.isEmpty()) {
                return failedFuture(new IllegalStateException("unknown leader location"));
            }

            if (HostUtils.getFqdn().equals(leaderHost.get())) {
                var assignments = balancer.getShardAssignmentsOrThrow();
                String[] hosts = balancer.getCluster().getHosts()
                    .stream()
                    .map(CoremonHost::getFqdn)
                    .toArray(String[]::new);

                var response = ShardAssignmentsSerializer.toResponse(hosts, assignments)
                    .toBuilder()
                    .setLeaderHost(HostUtils.getFqdn())
                    .build();
                return completedFuture(response);
            }

            final int ttl = request.getTtl();
            if (ttl == 0) {
                return failedFuture(new IllegalStateException("too many redirects"));
            }

            // redirect request to a leader host with TTL-1
            var redirectedRequest = request.toBuilder().setTtl(ttl - 1).build();
            CoremonHostClient peerClient = peers.get(leaderHost.get());
            return peerClient.getShardAssignments(redirectedRequest);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    CompletableFuture<TInitShardResponse> initShard(TInitShardRequest request) {
        Optional<String> leaderHost = balancer.getLeaderHost();
        if (leaderHost.isEmpty()) {
            return failedFuture(new IllegalStateException("unknown leader location"));
        }

        if (HostUtils.getFqdn().equals(leaderHost.get())) {
            // (1) assign shard to host
            return balancer.assignShard(request.getNumId())
                .thenComposeAsync(assignedToHost -> forceShardInit(assignedToHost, request.getProjectId(), request.getShardId())
                    .thenApply(ignore -> TInitShardResponse.newBuilder()
                        .setLeaderHost(HostUtils.getFqdn())
                        .setAssignedToHost(assignedToHost)
                        .build()));
        }

        final int ttl = request.getTtl();
        if (ttl == 0) {
            return failedFuture(new IllegalStateException("too many redirects"));
        }

        // redirect request to a leader host with TTL-1
        var redirectedRequest = request.toBuilder().setTtl(ttl - 1).build();
        CoremonHostClient peerClient = peers.get(leaderHost.get());
        return peerClient.initShard(redirectedRequest);
    }

    CompletableFuture<TCreateShardResponse> createShard(TCreateShardRequest createReq) {
        Optional<String> leaderHost = balancer.getLeaderHost();
        if (leaderHost.isEmpty()) {
            return failedFuture(new IllegalStateException("unknown leader location"));
        }

        if (HostUtils.getFqdn().equals(leaderHost.get())) {
            // (1) create shard config
            ShardKey shardKey = new ShardKey(createReq.getProjectId(), createReq.getClusterName(), createReq.getServiceName());
            return creator.createShard(shardKey, createReq.getCreatedBy())
                .thenCompose(shard -> {
                    // (2) assign shard to a host
                    return balancer.assignShard(shard.getNumId())
                        .thenApplyAsync(assignedToHost -> new Assignment(shard, assignedToHost));
                })
                .thenCompose(assignment -> {
                    String assignedToHost = assignment.host;
                    String projectId = assignment.shard.getProjectId();
                    String shardId = assignment.shard.getId();
                    int numId = assignment.shard.getNumId();

                    // (3) make sure that host loaded shard
                    return forceShardInit(assignedToHost, projectId, shardId)
                        .thenApply(ignore -> TCreateShardResponse.newBuilder()
                            .setLeaderHost(HostUtils.getFqdn())
                            .setAssignedToHost(assignedToHost)
                            .setShardId(shardId)
                            .setNumId(numId)
                            .build());
                });
        }

        final int ttl = createReq.getTtl();
        if (ttl == 0) {
            return failedFuture(new IllegalStateException("too many redirects"));
        }

        // redirect request to a leader host with TTL-1
        var redirectedRequest = createReq.toBuilder().setTtl(ttl - 1).build();
        CoremonHostClient peerClient = peers.get(leaderHost.get());
        return peerClient.createShard(redirectedRequest);
    }

    private CompletableFuture<?> forceShardInit(String shardLocation, String projectId, String shardId) {
        if (HostUtils.getFqdn().equals(shardLocation)) {
            return reloadShard(projectId, shardId, true, true);
        }

        var reloadRequest = TReloadShardRequest.newBuilder()
            .setProjectId(projectId)
            .setShardId(shardId)
            .setAwaitLoad(true)
            .setAllowUpdate(true)
            .build();

        // TODO: optimize this remote call, actual load can be done while assigning shard
        CoremonHostClient peer = peers.get(shardLocation);
        return peer.reloadShard(reloadRequest);
    }

    public CompletableFuture<String> deleteMetrics(DeleteMetricsRequest request) {
        try {
            logger.info(
                "{}deleting metrics ({}) {}/{} by {}",
                request.getInterrupt() ? "interrupt " : "",
                request.getPhase(),
                request.getOperationId(),
                Integer.toUnsignedString(request.getNumId()),
                request.getSelectors());

            return request.getInterrupt()
                ? interruptDeleteMetrics(request)
                : scheduleDeleteMetrics(request);
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<String> interruptDeleteMetrics(DeleteMetricsRequest request) {
        var taskId = createDeleteMetricsTaskId(request);
        return scheduler.getTask(taskId)
            .thenCompose(task -> {
                if (task.isEmpty()) {
                    return completedFuture("");
                }

                return scheduler.reschedule(
                        taskId,
                        System.currentTimeMillis(),
                        createDeleteMetricsInterruptOperator(request))
                    .thenApply(i -> taskId);
            });
    }

    private CompletableFuture<String> scheduleDeleteMetrics(DeleteMetricsRequest request) {
        var task = createDeleteMetricsTask(request);
        var taskId = task.id();
        return scheduler.schedule(task)
            .thenApply(ignore -> taskId);
    }

    private String createDeleteMetricsTaskId(DeleteMetricsRequest request) {
        var params = toDeleteMetricsParams(request);

        return switch (request.getPhase()) {
            case CHECK -> DeleteMetricsCheckTaskHandler.taskId(params);
            case MOVE -> DeleteMetricsMoveTaskHandler.taskId(params);
            case ROLLBACK -> DeleteMetricsRollbackTaskHandler.taskId(params);
            default -> throw Status.INVALID_ARGUMENT
                .withDescription("Unexpected delete metrics phase (interrupt): " + request.getPhase())
                .asRuntimeException();
        };
    }

    private ProgressOperator createDeleteMetricsInterruptOperator(DeleteMetricsRequest request) {
        return switch (request.getPhase()) {
            case CHECK -> DeleteMetricsCheckTaskHandler::interruptOperator;
            case MOVE -> DeleteMetricsMoveTaskHandler::interruptOperator;
            case ROLLBACK -> DeleteMetricsRollbackTaskHandler::interruptOperator;
            default -> throw Status.INVALID_ARGUMENT
                .withDescription("Unexpected delete metrics phase (interrupt): " + request.getPhase())
                .asRuntimeException();
        };
    }

    private Task createDeleteMetricsTask(DeleteMetricsRequest request) {
        var params = toDeleteMetricsParams(request);

        return switch (request.getPhase()) {
            case CHECK -> DeleteMetricsCheckTaskHandler.task(params);
            case MOVE -> DeleteMetricsMoveTaskHandler.task(params);
            case ROLLBACK -> DeleteMetricsRollbackTaskHandler.task(params);
            case TERMINATE -> DeleteMetricsTerminateTaskHandler.task(params);
            default -> throw Status.INVALID_ARGUMENT
                .withDescription("Unexpected delete metrics phase: " + request.getPhase())
                .asRuntimeException();
        };
    }

    private DeleteMetricsParams toDeleteMetricsParams(DeleteMetricsRequest request) {
        return DeleteMetricsParams.newBuilder()
            .setOperationId(request.getOperationId())
            .setNumId(request.getNumId())
            .setSelectors(request.getSelectors())
            .setCreatedAt(request.getCreatedAt())
            .setSubTaskCreatedAt(System.currentTimeMillis())
            .build();
    }

    @Override
    public void close() {
        peers.close();
    }

    private static record Assignment(Shard shard, String host) {
    }
}
