package ru.yandex.solomon.coremon.balancer;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.nio.file.Path;
import java.time.Clock;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.event.ContextRefreshedEvent;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.balancer.BalancerHolder;
import ru.yandex.solomon.balancer.BalancerHolderImpl;
import ru.yandex.solomon.balancer.TotalShardCounter;
import ru.yandex.solomon.balancer.TotalShardCounterImpl;
import ru.yandex.solomon.balancer.dao.BalancerDao;
import ru.yandex.solomon.balancer.dao.YdbBalancerDaoV2;
import ru.yandex.solomon.balancer.remote.RemoteCluster;
import ru.yandex.solomon.balancer.remote.RemoteClusterImpl;
import ru.yandex.solomon.balancer.remote.RemoteNodeClient;
import ru.yandex.solomon.balancer.remote.RemoteNodeClientImpl;
import ru.yandex.solomon.balancer.www.BalancerController;
import ru.yandex.solomon.config.protobuf.TKikimrClientConfig;
import ru.yandex.solomon.config.protobuf.coremon.CoremonBalancerConfig;
import ru.yandex.solomon.config.protobuf.http.HttpServerConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.coremon.CoremonDiscoveryContext;
import ru.yandex.solomon.coremon.balancer.db.BalancerShardsDao;
import ru.yandex.solomon.coremon.balancer.db.ydb.YdbBalancerShardsDao;
import ru.yandex.solomon.coremon.db.YdbClientsContext;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardPartition;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockService;
import ru.yandex.solomon.locks.LockServiceImpl;
import ru.yandex.solomon.locks.dao.LocksDao;
import ru.yandex.solomon.locks.dao.ydb.YdbLocksDao;
import ru.yandex.solomon.selfmon.mon.DaoMetricsProxy;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.util.host.HostUtils;

@Import({
        YdbClientsContext.class,
        CoremonDiscoveryContext.class,
})
@Configuration
public class MetabaseBalancerContext {

    private static final Path CACHE_DIR = Path.of("cache");

    private final MetricRegistry registry;
    private final Clock clock;
    private final ScheduledExecutorService timer;
    private final ExecutorService executor;
    private final TableClient tableClient;
    private final SchemeClient schemeClient;
    private final String schemeRoot;

    public MetabaseBalancerContext(
            CoremonBalancerConfig balancerThreadPoolConfig,
            MetricRegistry registry,
            Clock clock,
            ThreadPoolProvider threadPoolProvider,
            @Qualifier("localTableClient") TableClient tableClient,
            @Qualifier("localSchemeClient") SchemeClient schemeClient,
            @Qualifier("KikimrClientConfig") TKikimrClientConfig kikimrClientConfig)
    {
        this.registry = registry;
        this.clock = clock;
        this.timer = threadPoolProvider.getSchedulerExecutorService();
        this.tableClient = tableClient;
        this.schemeClient = schemeClient;
        this.schemeRoot = kikimrClientConfig.getSchemaRoot() + "/V1";
        this.executor = threadPoolProvider.getExecutorService(
                balancerThreadPoolConfig.getThreadPoolName(),
                "CoremonBalancerConfig.ThreadPoolName");
    }

    @Bean
    LocksDao locksDao() {
        var dao = new YdbLocksDao(tableClient, clock, schemeRoot);
        return measure(dao, LocksDao.class);
    }

    @Bean
    public BalancerDao balancerDao() {
        var dao = new YdbBalancerDaoV2(schemeRoot, tableClient, schemeClient);
        dao.createSchema();
        return measure(dao, BalancerDao.class);
    }

    @Bean
    public BalancerShardsDao balancerShardsDao() {
        var dao = new YdbBalancerShardsDao(schemeRoot, tableClient, schemeClient);
        dao.createSchemaForTests();
        return measure(dao, BalancerShardsDao.class);
    }

    private <TDao> TDao measure(TDao dao, Class<TDao> daoClass) {
        return DaoMetricsProxy.of(dao, daoClass, registry);
    }

    @Bean
    public LockService lockService(LocksDao dao) {
        return new LockServiceImpl(HostUtils.getFqdn(), dao, clock, timer, registry);
    }

    @Bean
    @BalancerLock
    public DistributedLock balancerLock(LockService lockService) {
        return lockService.distributedLock("MetabaseBalancer");
    }

    @Bean
    public TotalShardCounter totalShardPartitionCounter() {
        return new TotalShardCounterImpl();
    }

    @Bean
    public RemoteNodeClient remoteNodeClient(
            ClusterDiscovery<GrpcTransport> discovery,
            TotalShardCounter totalShardPartitionCounter)
    {
        return new RemoteNodeClientImpl(discovery, totalShardPartitionCounter);
    }

    @Bean
    public RemoteCluster remoteCluster(RemoteNodeClient remoteNodeClient) {
        return new RemoteClusterImpl(remoteNodeClient, timer, clock);
    }

    @Bean
    public MetabaseShardPartitionsHolder shardPartitionsHolder(
            BalancerShardsDao dao,
            @BalancerLock DistributedLock lock)
    {
        return new MetabaseShardPartitionsHolderImpl(dao, lock);
    }

    @Bean
    public BalancerHolderImpl balancerHolder(
            MetabaseShardPartitionsHolder metabaseShardPartitionsHolder,
            TotalShardCounter totalShardPartitionCounter,
            RemoteCluster cluster,
            DistributedLock lock,
            BalancerDao balancerDao)
    {
        return new BalancerHolderImpl(
                clock,
                lock,
                MetabaseResource.RESOURCES,
                cluster,
                executor,
                timer,
                metabaseShardPartitionsHolder,
                totalShardPartitionCounter,
                balancerDao);
    }

    @Bean
    public Bootstrap balancerBootstrap(BalancerHolderImpl balancer) {
        return new Bootstrap(balancer);
    }

    @Bean
    public BalancerController balancerController(
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer,
            BalancerHolder balancerHolder,
            HttpServerConfig config)
    {
        return new BalancerController(
                authenticator,
                authorizer,
                balancerHolder,
                MetabaseShardPartition.class,
                config.getPort()
        );
    }

    @Bean
    public RootLink balancerLink() {
        return new RootLink("/balancer", "Balancer");
    }

    @Qualifier
    @Retention(RetentionPolicy.RUNTIME)
    public @interface BalancerLock {
    }

    @ParametersAreNonnullByDefault
    private static class Bootstrap implements ApplicationListener<ContextRefreshedEvent> {
        private final BalancerHolderImpl balancer;

        Bootstrap(BalancerHolderImpl balancer) {
            this.balancer = balancer;
        }

        @Override
        public void onApplicationEvent(ContextRefreshedEvent event) {
            balancer.start();
        }
    }
}
