package ru.yandex.solomon.coremon.balancer;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import io.grpc.Status;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import it.unimi.dsi.fastutil.ints.IntSets;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.balancer.Balancer;
import ru.yandex.solomon.balancer.BalancerHolderImpl;
import ru.yandex.solomon.balancer.NodeStatus;
import ru.yandex.solomon.balancer.NodeSummary;
import ru.yandex.solomon.core.conf.ShardConfMaybeWrong;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.watch.SolomonConfListener;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.coremon.balancer.Routines.AssignLostShards;
import ru.yandex.solomon.coremon.balancer.Routines.AssignNewShard;
import ru.yandex.solomon.coremon.balancer.Routines.KickShard;
import ru.yandex.solomon.coremon.balancer.Routines.MoveShards;
import ru.yandex.solomon.coremon.balancer.Routines.Rebalance;
import ru.yandex.solomon.coremon.balancer.cluster.CoremonCluster;
import ru.yandex.solomon.coremon.balancer.cluster.CoremonHost;
import ru.yandex.solomon.coremon.balancer.db.ShardAssignments;
import ru.yandex.solomon.coremon.balancer.db.ShardAssignmentsDao;
import ru.yandex.solomon.coremon.balancer.db.ShardBalancerOptions;
import ru.yandex.solomon.coremon.balancer.db.ShardBalancerOptionsDao;
import ru.yandex.solomon.coremon.meta.service.MetabaseTotalShardCounter;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockDetail;
import ru.yandex.solomon.locks.LockSubscriber;
import ru.yandex.solomon.locks.UnlockReason;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.util.host.HostUtils;
import ru.yandex.solomon.util.time.DurationUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static com.google.common.util.concurrent.MoreExecutors.shutdownAndAwaitTermination;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.misc.concurrent.CompletableFutures.join;

/**
 *
 *                                             +--------------------+
 *                                             |                    |
 *                                         +-->+ Local Coremon Host |
 *                                         |   |                    |
 * +------------------+    +-----------+   |   +--------------------+
 * |                  |    |           |   |
 * |  Shard Balancer  +--->+  Cluster  +---+
 * |                  |    |           |   |       +---------------------+
 * +------------------+    +-----------+   |     +---------------------+ |
 *                                         |   +---------------------+ | |
 *                                         |   |                     | | |
 *                                         +-->+ Remote Coremon Host | +-+
 *                                             |                     +-+
 *                                             +---------------------+
 *                    ~~~~~~~~[network]~~~~~~~~~~~~~~~~~~|~|~|~~~~~~~~~~~~~~~~~~~~~~~~~~~~[network]~~~~~~~~~
 *                                                       v v v
 *                                                   +----------------+          +--------------------+
 *                                                 +----------------+ |        +--------------------+ |
 *                                               +----------------+ | |      +--------------------+ | |
 *                                               |                | | | ---> |                    | | |
 *                                               | Remote Coremon | | | ---> | Local Coremon Host | +-+
 *                                               |    Host Peer   | +-+ ---> |                    +-+
 *                                               |                +-+        +--------------------+
 *                                               +----------------+
 *
 * @author Sergey Polovko
 */
public class ShardBalancer implements SolomonConfListener, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ShardBalancer.class);

    private final ShardBalancerMetrics metrics;
    private final CoremonCluster cluster;
    private final DistributedLock lock;
    private final ShardAssignmentsDao assignmentsDao;
    private final ShardBalancerOptionsDao balancerOptionsDao;
    private final BalancerHolderImpl newBalancerHolder;
    private final ScheduledExecutorService worker;

    private BalancerMode activeBalancerMode = BalancerMode.NONE;
    private ShardBalancerOptions options = ShardBalancerOptions.DEFAULT;
    private boolean holderInitialized;

    private volatile boolean newBalancerInitialized;
    private final AtomicReference<ShardAssignments> newBalancerAssignments = new AtomicReference<>(ShardAssignments.EMPTY);

    private volatile boolean stopped = false;
    private volatile SolomonConfWithContext lastConf;
    private final AtomicReference<ShardAssignments> assignments = new AtomicReference<>(ShardAssignments.EMPTY);
    private final AtomicInteger totalShardCount = new AtomicInteger(MetabaseTotalShardCounter.SHARD_COUNT_UNKNOWN);
    private volatile boolean reloadAssignments = true;
    private volatile long becomeLeaderAt = 0;
    private final CoremonShardsHolder shardsHolder;

    // for manager ui
    private Throwable lastError;
    private Instant lastErrorTime = Instant.EPOCH;


    public ShardBalancer(
        CoremonCluster cluster,
        DistributedLock lock,
        ShardAssignmentsDao assignmentsDao,
        ShardBalancerOptionsDao balancerOptionsDao,
        BalancerHolderImpl newBalancerHolder,
        CoremonShardsHolder shardsHolder,
        MetricRegistry metricRegistry)
    {
        this.metrics = new ShardBalancerMetrics(metricRegistry);
        this.cluster = cluster;
        this.lock = lock;
        this.assignmentsDao = assignmentsDao;
        this.balancerOptionsDao = balancerOptionsDao;
        this.newBalancerHolder = newBalancerHolder;
        this.shardsHolder = shardsHolder;

        // there is no more important work in Coremon than being responsible to balance shards
        // so we spend whole one thread for doing this
        this.worker = Executors.newSingleThreadScheduledExecutor(r -> new Thread(r, "ShardBalancerThread"));
    }

    public void start() {
        this.worker.execute(this::loop);
    }

    private void loop() {
        if (!stopped) {
            long delayMillis = iteration();
            worker.schedule(this::loop, DurationUtils.randomize(delayMillis), TimeUnit.MILLISECONDS);
        }
    }

    /**
     * @return delay in milliseconds before next iteration
     */
    private long iteration() {
        try {
            clearLastError();

            if (lastConf == null) {
                // await first config update
                return 1_000;
            }

            options = join(balancerOptionsDao.load());
            switch (activeBalancerMode) {
                case NONE -> {
                    if (stopped) {
                        return 0;
                    }

                    if (options.isUseNewBalancer()) {
                        newBalancerHolder.start();
                        activeBalancerMode = BalancerMode.NEW;
                    } else {
                        tryAcquireLeadership();
                        activeBalancerMode = BalancerMode.OLD;
                    }

                    logger.info("starting in balancer mode: {}", activeBalancerMode);
                    return 1_000;
                }

                case OLD -> {
                    if (options.isUseNewBalancer()) {
                        logger.info("switching balancer mode: {} -> {}", BalancerMode.OLD, BalancerMode.NEW);

                        join(lock.unlock());
                        reloadAssignments = true;
                        holderInitialized = false;
                        metrics.reset();
                        assignments.set(ShardAssignments.EMPTY);
                        totalShardCount.set(MetabaseTotalShardCounter.SHARD_COUNT_UNKNOWN);

                        newBalancerHolder.start();

                        activeBalancerMode = BalancerMode.NEW;
                        return 1_000;
                    } else {
                        // continue the good old iteration
                    }
                }

                case NEW -> {
                    if (!options.isUseNewBalancer()) {
                        logger.info("switching balancer mode: {} -> {}", BalancerMode.NEW, BalancerMode.OLD);

                        newBalancerHolder.stop();
                        newBalancerInitialized = false;
                        newBalancerAssignments.set(ShardAssignments.EMPTY);

                        tryAcquireLeadership();

                        activeBalancerMode = BalancerMode.OLD;
                        return 1_000;
                    }

                    actualizeNewBalancerAssignments();
                    return 1_000;
                }
            }

            Optional<LockDetail> lockDetail = lock.lockDetail();
            if (lockDetail.isEmpty()) {
                logger.info("no information about who is leader");
                return 1_000;
            }

            if (!lock.isLockedByMe()) {
                // clean up previous state if not a leader, instead of clean up it during running
                // leader routine, that can lead to reassign all shards
                reloadAssignments = true;
                holderInitialized = false;
                metrics.reset();
                assignments.set(ShardAssignments.EMPTY);
                totalShardCount.set(MetabaseTotalShardCounter.SHARD_COUNT_UNKNOWN);
                if (logger.isDebugEnabled()) {
                    logger.debug("I'm not a leader, current leader is: {}", lockDetail.get().owner());
                }
                return 1_000;
            }

            leaderRoutine();
            metrics.iterationOk.inc();

            return 5_000;
        } catch (Throwable t) {
            logger.error("unhandled exception in shard balancer loop", t);
            lastError = t;
            lastErrorTime = Instant.now();
            metrics.iterationFail.inc();
            return 5_000;
        }
    }

    public CompletableFuture<Void> kickShardsFrom(String fqdn) {
        return CompletableFuture.runAsync(() -> {
            throwIfNewBalancerIsActive();

            final CoremonHost host = Objects.requireNonNull(cluster.getHost(fqdn));
            CoremonCluster.Hosts hosts = cluster.getHosts(options.getOfflineThresholdMillis());
            var onlineHosts = filterHosts(hosts.getOnline(), options.getInactiveHosts());

            var moveShards = new MoveShards(onlineHosts, assignmentsDao);
            mergeAssignments(moveShards.run(host));
        }, worker);
    }

    public CompletableFuture<String> assignShard(int shardNumId) {
        // fastpass check
        {
            String assignedHost = assignments.get().get(shardNumId);
            if (assignedHost != null) {
                return completedFuture(assignedHost);
            }
        }

        var resultFuture = new CompletableFuture<String>();
        CompletableFuture.runAsync(() -> {
            try {
                if (activeBalancerMode == BalancerMode.NEW) {
                    var shardId = Integer.toUnsignedString(shardNumId);
                    var assignmentFuture = getNewBalancerOrThrow().getOrCreateAssignment(shardId);
                    CompletableFutures.whenComplete(assignmentFuture, resultFuture);
                    return;
                }

                // check once again to avoid races
                var assignments = this.assignments.get();
                String assignedHost = assignments.get(shardNumId);
                if (assignedHost != null) {
                    resultFuture.complete(assignedHost);
                    return;
                }

                CoremonCluster.Hosts hosts = cluster.getHosts(options.getOfflineThresholdMillis());
                var onlineHosts = filterHosts(hosts.getOnline(), options.getInactiveHosts());

                var r = new AssignNewShard(assignmentsDao, onlineHosts, options);
                var host = join(r.run(shardNumId));

                shardsHolder.add(shardNumId).join();
                var update = ShardAssignments.singleton(shardNumId, host);
                mergeAssignments(update);
                metrics.addShards.add(1);

                resultFuture.complete(host);
            } catch (Throwable t) {
                resultFuture.completeExceptionally(t);
            }
        }, worker);
        return resultFuture;
    }

    public CompletableFuture<Void> moveShard(String fqdn, Integer numId, String strId) {
        return CompletableFuture.runAsync(() -> {
            throwIfNewBalancerIsActive();

            Set<String> knownFqdns = cluster.getHosts().stream()
                .map(CoremonHost::getFqdn)
                .collect(Collectors.toSet());
            checkState(knownFqdns.contains(fqdn), "invalid FQDN: %s", fqdn);

            final ShardAssignments assign;
            if (numId != null) {
                ShardConfMaybeWrong shardConf = lastConf.getShardByNumIdOrNull(numId);
                Preconditions.checkNotNull(shardConf, "unknown shard num id: %s", Integer.toUnsignedLong(numId));
                assign = ShardAssignments.singleton(numId, fqdn);
            } else {
                ShardConfMaybeWrong shardConf = lastConf.getShardByIdOrNull(strId);
                Preconditions.checkNotNull(shardConf, "unknown shard str id: %s", strId);
                assign = ShardAssignments.singleton(shardConf.getNumId(), fqdn);
            }

            logger.info("moving shard {} to host {}", Integer.toUnsignedLong(assign.getIds().iterator().nextInt()), fqdn);
            assignmentsDao.save(assign).join();
            cluster.updateAssignments(mergeAssignments(assign));
        }, worker);
    }

    public CompletableFuture<Void> kickShard(String shardId, boolean allowNewBalancer) {
        var doneFuture = new CompletableFuture<Void>();
        CompletableFuture.runAsync(() -> {
            try {
                if (!allowNewBalancer) {
                    throwIfNewBalancerIsActive();
                } else {
                    if (activeBalancerMode == BalancerMode.NEW) {
                        var kickFuture = getNewBalancerOrThrow().kickShard(shardId).thenRun(() -> {});
                        CompletableFutures.whenComplete(kickFuture, doneFuture);
                        return;
                    }
                }

                ShardConfMaybeWrong shardConf = lastConf.getShardByIdOrNull(shardId);
                if (shardConf == null) {
                    shardConf = lastConf.getShardByNumId(Integer.parseUnsignedInt(shardId));
                }

                Preconditions.checkNotNull(shardConf, "unknown shard: %s", shardId);
                int numId = shardConf.getNumId();
                var fqnd = getShardAssignmentsOrThrowOld().get(numId);

                CoremonCluster.Hosts hosts = cluster.getHosts(options.getOfflineThresholdMillis());
                var onlineHosts = filterHosts(hosts.getOnline(), options.getInactiveHosts());

                logger.info("kick shard {} from host {}", Integer.toUnsignedString(numId), fqnd);
                KickShard kickShard = new KickShard(assignmentsDao, onlineHosts, options);
                cluster.updateAssignments(mergeAssignments(kickShard.run(numId).join()));

                doneFuture.complete(null);
            } catch (Throwable t) {
                doneFuture.completeExceptionally(t);
            }
        }, worker);
        return doneFuture;
    }

    private void leaderRoutine() {
        SolomonConfWithContext lastConf = this.lastConf;
        ShardAssignments assignments = this.assignments.get();

        if (!holderInitialized) {
            shardsHolder.init(lastConf).join();
            holderInitialized = true;
        }

        // (0) force assignments synchronization with peers
        //     do it before any other actions to prevent synchronization stuck
        //     due to some other errors
        if (!reloadAssignments) {
            cluster.updateAssignments(assignments);
        }

        // (1) restore assignments
        if (assignments.isEmpty() || reloadAssignments) {
            logger.info("assignments set is empty, reload it");
            var loaded = join(assignmentsDao.load());
            logger.info("loaded {} assignments of {} known shard", loaded.size(), lastConf.getAllShardsCount());
            assignments = mergeAssignments(loaded);
            reloadAssignments = false;
            // update assignments on cluster after reload it
            cluster.updateAssignments(assignments);
        }

        // (2) delete assignments for extinct shards
        IntSet toDelete = deleteShards(lastConf, assignments);
        if (!toDelete.isEmpty()) {
            metrics.deleteShards.add(toDelete.size());
            logger.info("delete assignments for shards: {}", IntStream.of(toDelete.toIntArray())
                    .mapToObj(Integer::toUnsignedLong)
                    .collect(Collectors.toList()));
            join(assignmentsDao.delete(toDelete));
            assignments = updateAssignments(current -> current.delete(toDelete));
        }

        // (3) find offline and online hosts
        CoremonCluster.Hosts hosts = cluster.getHosts(options.getOfflineThresholdMillis());
        var onlineHosts = filterHosts(hosts.getOnline(), options.getInactiveHosts());
        metrics.hostsOffline.set(hosts.getOffline().size());
        metrics.hostsOnline.set(onlineHosts.size());

        if (onlineHosts.size() < hosts.getOffline().size()) {
            logger.warn("skip shads moving because too many offline hosts, offline ({}): {}, online ({}): {}",
                hosts.getOffline().size(), hosts.getOffline(),
                onlineHosts.size(), onlineHosts);
            return;
        }

        // (4) move shards from offline host, only if there are more online hosts than offline
        if (!hosts.getOffline().isEmpty()) {
            long leaderTimeMillis = System.currentTimeMillis() - becomeLeaderAt;
            if (leaderTimeMillis > TimeUnit.MINUTES.toMillis(5)) {
                var moveShards = new MoveShards(onlineHosts, assignmentsDao);
                for (CoremonHost host : hosts.getOffline()) {
                    assignments = mergeAssignments(moveShards.run(host));
                }
            } else {
                List<String> offlineFqdns = hosts.getOffline().stream()
                    .map(CoremonHost::getFqdn)
                    .collect(Collectors.toList());
                logger.warn(
                    "found offline hosts {}, but leader is too young {}ms to move shards from them",
                    offlineFqdns, leaderTimeMillis);
            }
        }

        // (5) assign not assigned shards
        {
            var assignLostShards = new AssignLostShards(lastConf, onlineHosts, assignmentsDao, options);
            var lost = assignLostShards.run(assignments);
            metrics.addShards.add(lost.size());
            assignments = mergeAssignments(lost);
        }

        // (6) rebalance shards
        if (options.getRebalaceShardsInFlight() > 0) {
            var rebalance = new Rebalance(onlineHosts, assignmentsDao, metrics);
            mergeAssignments(rebalance.run(assignments, options));
        }
    }

    @NotNull
    private IntSet deleteShards(SolomonConfWithContext lastConf, ShardAssignments assignments) {
        IntSet toDelete = new IntOpenHashSet(assignments.getIds());
        for (Shard s : lastConf.getAllRawShards()) {
            toDelete.remove(s.getNumId());
        }

        if (toDelete.isEmpty()) {
            return IntSets.EMPTY_SET;
        }

        // skip delete recently created shards
        var it = toDelete.iterator();
        while (it.hasNext()) {
            int numId = it.nextInt();
            if (shardsHolder.recentlyAdded(numId)) {
                it.remove();
            }
        }

        // delete assignments from not exists nodes
        for (var entry : assignments.reverse().entrySet()) {
            if (cluster.getHost(entry.getKey()) == null) {
                toDelete.addAll(entry.getValue());
            }
        }

        return toDelete;
    }

    private ShardAssignments updateAssignments(Function<ShardAssignments, ShardAssignments> fn) {
        ShardAssignments prev;
        ShardAssignments next;
        do {
            if (!isLeader()) {
                throw new IllegalStateException(HostUtils.getShortName() + " not leader anymore");
            }
            prev = this.assignments.get();
            next = fn.apply(prev);
        } while (!this.assignments.compareAndSet(prev, next));
        this.totalShardCount.set(next.size());
        this.metrics.shards.set(next.size());
        return next;
    }

    private ShardAssignments mergeAssignments(ShardAssignments delta) {
        return updateAssignments(assignments -> assignments.mergeWith(delta));
    }

    private static List<CoremonHost> filterHosts(List<CoremonHost> hosts, Set<String> inactive) {
        var builder = ImmutableList.<CoremonHost>builder();
        for (CoremonHost host : hosts) {
            if (!inactive.contains(host.getFqdn())) {
                builder.add(host);
            }
        }
        return builder.build();
    }

    private void tryAcquireLeadership() {
        if (stopped) {
            return;
        }

        lock.acquireLock(new LockSubscriber() {
            @Override
            public boolean isCanceled() {
                return stopped;
            }

            @Override
            public void onLock(long seqNo) {
                cluster.startPinging(seqNo, totalShardCount::get);
                metrics.isLeader.set(1);
                becomeLeaderAt = System.currentTimeMillis();
                reloadAssignments = true;
                logger.info("{} become leader at {}", HostUtils.getShortName(), becomeLeaderAt);
            }

            @Override
            public void onUnlock(UnlockReason reason) {
                cluster.stopPinging();
                metrics.reset();
                logger.info("{} become member", HostUtils.getShortName());
                if (reason != UnlockReason.MANUAL_UNLOCK) {
                    tryAcquireLeadership();
                }
            }
        }, 60_000, TimeUnit.MILLISECONDS);
    }

    private void clearLastError() {
        if (lastErrorTime != Instant.EPOCH) {
            var elapsed = Duration.between(Instant.now(), lastErrorTime).abs();
            if (elapsed.compareTo(Duration.ofHours(7)) >= 0) {
                lastErrorTime = Instant.EPOCH;
                lastError = null;
            }
        }
    }

    @Override
    public void close() {
        stopped = true;
        shutdownAndAwaitTermination(worker, 10, TimeUnit.SECONDS);
        lock.unlock().join();
        newBalancerHolder.close();
    }

    @Override
    public void onConfigurationLoad(SolomonConfWithContext conf) {
        lastConf = conf;
    }

    private boolean isLeader() {
        return lock.isLockedByMe();
    }

    public boolean isLeaderOnThisHost() {
        return getLeaderHost()
            .map(leaderHost -> HostUtils.getFqdn().equals(leaderHost))
            .orElse(false);
    }

    public Optional<String> getLeaderHost() {
        return lock.lockDetail()
            .map(lockDetail -> {
                var owner = lockDetail.owner();
                if (owner == null) {
                    return null;
                }

                if (owner.endsWith("_new") || owner.endsWith("_old")) {
                    return owner.substring(0, owner.length() - 4);
                }

                return owner;
            });
    }

    public CoremonCluster getCluster() {
        return cluster;
    }

    @ManagerMethod
    void reloadAssignments() {
        var f = CompletableFuture.runAsync(() -> {
            throwIfNewBalancerIsActive();

            var newAssignments = join(assignmentsDao.load());
            checkArgument(!newAssignments.isEmpty(), "load empty assignments");
            this.assignments.set(newAssignments);
            this.metrics.shards.set(newAssignments.size());
            cluster.updateAssignments(newAssignments);
        });

        join(f);
    }

    /**
     * @return map {shardId -> host}
     */
    public Int2ObjectMap<String> getShardAssignmentsOrThrow() {
        if (newBalancerInitialized) {
            return newBalancerAssignments.get().asMap();
        }
        return getShardAssignmentsOrThrowOld();
    }

    private Int2ObjectMap<String> getShardAssignmentsOrThrowOld() {
        ShardAssignments copy;
        do {
            copy = this.assignments.get();
            if (!copy.isEmpty()) {
                return copy.asMap();
            }

            if (this.reloadAssignments) {
                throw new StatusRuntimeExceptionNoStackTrace(Status.UNAVAILABLE
                        .withDescription("shard assignments not initialized yet"));
            }
        } while (!copy.equals(this.assignments.get()));
        return copy.asMap();
    }

    private void actualizeNewBalancerAssignments() {
        var newBalancer = newBalancerHolder.getBalancer();
        if (newBalancer == null) {
            // not a leader, clean up the previous state
            this.newBalancerInitialized = false;
            this.newBalancerAssignments.set(ShardAssignments.EMPTY);
            return;
        }

        var initialized = this.newBalancerInitialized;
        if (initialized) {
            this.newBalancerAssignments.set(computeNewBalancerAssignments(newBalancer));
        } else {
            if (!newBalancer.getNodes().isEmpty()) {
                initialized = newBalancer.getNodes().values().stream()
                    .filter(NodeSummary::isActive)
                    .allMatch(node -> node.getStatus() != NodeStatus.UNKNOWN);
            }

            if (initialized) {
                this.newBalancerAssignments.set(computeNewBalancerAssignments(newBalancer));
                this.newBalancerInitialized = true;
            }
        }
    }

    private ShardAssignments computeNewBalancerAssignments(Balancer newBalancer) {
        var shards = newBalancer.getShards().values();

        var shard2host = new Int2ObjectOpenHashMap<String>(shards.size());
        for (var shard : shards) {
            var shardNode = shard.getNode();
            if (shardNode != null) {
                shard2host.put(Integer.parseUnsignedInt(shard.getShardId()), shardNode.getAddress());
            }
        }

        return ShardAssignments.ownOf(shard2host);
    }

    private void throwIfNewBalancerIsActive() {
        if (activeBalancerMode == BalancerMode.NEW) {
            throw new IllegalStateException("old is not active, use new balancer ctl instead");
        }
    }

    private Balancer getNewBalancerOrThrow() {
        var newBalancer = newBalancerHolder.getBalancer();
        if (newBalancer == null) {
            throw new IllegalStateException("unknown leader location");
        }
        return newBalancer;
    }

    private enum BalancerMode {
        NONE,
        OLD,
        NEW
    }
}
