package ru.yandex.solomon.coremon.balancer;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.time.Clock;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.event.ContextRefreshedEvent;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.balancer.BalancerHolder;
import ru.yandex.solomon.balancer.BalancerHolderImpl;
import ru.yandex.solomon.balancer.ShardsHolder;
import ru.yandex.solomon.balancer.TotalShardCounter;
import ru.yandex.solomon.balancer.TotalShardCounterImpl;
import ru.yandex.solomon.balancer.dao.BalancerDao;
import ru.yandex.solomon.balancer.dao.YdbBalancerDaoV2;
import ru.yandex.solomon.balancer.remote.RemoteCluster;
import ru.yandex.solomon.balancer.remote.RemoteClusterImpl;
import ru.yandex.solomon.balancer.remote.RemoteNodeClient;
import ru.yandex.solomon.balancer.remote.RemoteNodeClientImpl;
import ru.yandex.solomon.balancer.www.BalancerController;
import ru.yandex.solomon.config.protobuf.TKikimrClientConfig;
import ru.yandex.solomon.config.protobuf.coremon.CoremonBalancerConfig;
import ru.yandex.solomon.config.protobuf.coremon.TCoremonBalancerConfig;
import ru.yandex.solomon.config.protobuf.http.HttpServerConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.coremon.CoremonDiscoveryContext;
import ru.yandex.solomon.coremon.CoremonShard;
import ru.yandex.solomon.coremon.CoremonState;
import ru.yandex.solomon.coremon.balancer.cluster.CoremonCluster;
import ru.yandex.solomon.coremon.balancer.cluster.GrpcBalancerService;
import ru.yandex.solomon.coremon.balancer.cluster.LocalCoremonHost;
import ru.yandex.solomon.coremon.balancer.cluster.RemoteCoremonHostPeer;
import ru.yandex.solomon.coremon.balancer.db.BalancerShardsDao;
import ru.yandex.solomon.coremon.balancer.db.CoremonMigrateBalancerDao;
import ru.yandex.solomon.coremon.balancer.db.ShardAssignmentsDao;
import ru.yandex.solomon.coremon.balancer.db.ShardBalancerOptionsDao;
import ru.yandex.solomon.coremon.balancer.db.YdbShardAssignmentsDao;
import ru.yandex.solomon.coremon.balancer.db.YdbShardBalancerOptionsDao;
import ru.yandex.solomon.coremon.balancer.db.ydb.YdbBalancerShardsDao;
import ru.yandex.solomon.coremon.balancer.www.ShardBalancerController;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockService;
import ru.yandex.solomon.locks.LockServiceImpl;
import ru.yandex.solomon.locks.dao.LocksDao;
import ru.yandex.solomon.locks.dao.ydb.YdbLocksDao;
import ru.yandex.solomon.selfmon.mon.DaoMetricsProxy;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Sergey Polovko
 */
@Import({
    CoremonDiscoveryContext.class
})
@Configuration
public class ShardBalancerContext {

    private final MetricRegistry registry;
    private final Clock clock;
    private final ScheduledExecutorService timer;
    private final ExecutorService executor;

    private final TableClient tableClient;
    private final SchemeClient schemeClient;

    private final String schemeRoot;

    @Autowired
    public ShardBalancerContext(
        MetricRegistry registry,
        Clock clock,
        ThreadPoolProvider threads,
        CoremonBalancerConfig balancerConfig,
        @Qualifier("localTableClient") TableClient tableClient,
        @Qualifier("localSchemeClient") SchemeClient schemeClient,
        @Qualifier("KikimrClientConfig") TKikimrClientConfig kikimrClientConfig)
    {
        this.registry = registry;
        this.clock = clock;
        this.timer = threads.getSchedulerExecutorService();
        this.executor = threads.getExecutorService(
            balancerConfig.getThreadPoolName(),
            "CoremonBalancerConfig.ThreadPoolName");

        this.tableClient = tableClient;
        this.schemeClient = schemeClient;

        this.schemeRoot = kikimrClientConfig.getSchemaRoot() + "/V1";
    }

    @Bean
    public TotalShardCounter totalShardCounter() {
        return new TotalShardCounterImpl();
    }

    @Bean
    public RemoteNodeClient remoteNodeClient(
        ClusterDiscovery<GrpcTransport> discovery,
        TotalShardCounter shardCounter)
    {
        return new RemoteNodeClientImpl(discovery, shardCounter);
    }

    @Bean
    public RemoteCluster remoteCluster(RemoteNodeClient remoteNodeClient) {
        return new RemoteClusterImpl(remoteNodeClient, timer, clock);
    }

    @Bean
    public CoremonShardsHolder shardsHolder(BalancerShardsDao dao, @NewBalancer DistributedLock lock) {
        return new CoremonShardsHolder(dao, lock);
    }

    @Bean
    public BalancerShardsDao balancerShardsDao() {
        var dao = new YdbBalancerShardsDao(schemeRoot, tableClient, schemeClient);
        dao.createSchemaForTests(); //TODO: remove after migration <SOLOMON-6713>
        return measure(dao, BalancerShardsDao.class);
    }

    @Bean(destroyMethod = "")
    public BalancerHolderImpl balancerHolder(
        @NewBalancer DistributedLock lock,
        RemoteCluster cluster,
        ShardsHolder shards,
        TotalShardCounter shardCounter,
        BalancerDao dao)
    {
        return new CoremonBalancerHolder(
            clock,
            lock,
            CoremonResource.RESOURCES,
            cluster,
            executor,
            timer,
            shards,
            shardCounter,
            dao);
    }

    @Bean
    public Bootstrap balancerBootstrap(ShardBalancer balancer) {
        return new Bootstrap(balancer);
    }

    @Bean
    public BalancerController balancerController(
        HttpAuthenticator authenticator,
        InternalAuthorizer authorizer,
        BalancerHolder balancer,
        HttpServerConfig config)
    {
        return new BalancerController(authenticator, authorizer, balancer, CoremonShard.class, config.getPort());
    }

    @Deprecated
    @Bean
    ShardLocatorImpl shardLocator() {
        return new ShardLocatorImpl();
    }

    @Deprecated
    @Bean
    LocalCoremonHost localHost(CoremonState coremonState, ThreadPoolProvider threadPoolProvider) {
        return new LocalCoremonHost(coremonState, threadPoolProvider.getSchedulerExecutorService());
    }

    @Deprecated
    @Bean
    RemoteCoremonHostPeer remotePeer(LocalCoremonHost localHost) {
        return new RemoteCoremonHostPeer(localHost);
    }

    @Deprecated
    @Bean
    CoremonCluster coremonCluster(
        @Qualifier("ClientId") Optional<String> clientId,
        TCoremonBalancerConfig config,
        ThreadPoolProvider threadPoolProvider,
        LocalCoremonHost localHost,
        ClientOptionsFactory clientOptionsFactory)
    {
        return new CoremonCluster(clientId.orElse("coremon"), config, threadPoolProvider, localHost, clientOptionsFactory);
    }

    @Deprecated
    @Bean
    ShardBalancer balancerState(
        CoremonCluster cluster,
        @OldBalancer DistributedLock lock,
        ShardAssignmentsDao assignmentsDao,
        ShardBalancerOptionsDao balancerOptionsDao,
        CoremonShardsHolder shardsHolder,
        BalancerHolderImpl newBalancerHolder)
    {
        return new ShardBalancer(
            cluster,
            lock,
            assignmentsDao,
            balancerOptionsDao,
            newBalancerHolder,
            shardsHolder,
            registry);
    }

    @Deprecated
    @Bean
    ShardBalancerController oldBalancerController(
        HttpAuthenticator authenticator,
        InternalAuthorizer authorizer,
        ShardBalancer balancer,
        ShardBalancerOptionsDao balancerOptionsDao,
        HttpServerConfig config)
    {
        int httpInternalPort = config.getPort();
        return new ShardBalancerController(authenticator, authorizer, balancer, balancerOptionsDao, httpInternalPort);
    }

    @Bean
    @OldBalancer
    public DistributedLock oldBalancerLock(@OldBalancer LockService lockService) {
        return lockService.distributedLock("ShardBalancer", "OldShardBalancer");
    }

    @Bean
    @NewBalancer
    public DistributedLock balancerLock(@NewBalancer LockService lockService) {
        return lockService.distributedLock("ShardBalancer");
    }

    @Deprecated
    @Bean
    @OldBalancer
    LockService oldLockService(LocksDao dao) {
        return new LockServiceImpl(HostUtils.getFqdn() + "_old", dao, clock, timer, registry);
    }

    @Bean
    @NewBalancer
    LockService lockService(LocksDao dao) {
        return new LockServiceImpl(HostUtils.getFqdn() + "_new", dao, clock, timer, registry);
    }

    @Bean
    LocksDao locksDao() {
        var dao = new YdbLocksDao(tableClient, clock, schemeRoot);
        return measure(dao, LocksDao.class);
    }

    @Bean
    public BalancerDao balancerDao(ShardAssignmentsDao assignmentsDao) {
        var dao = new YdbBalancerDaoV2(schemeRoot, tableClient, schemeClient);
        dao.createSchema(); //TODO: remove after migration <SOLOMON-6713>
        var balancerDao = measure(dao, BalancerDao.class);

        return new CoremonMigrateBalancerDao(assignmentsDao, balancerDao);
    }

    @Deprecated
    @Bean
    ShardAssignmentsDao assignmentsDao() {
        var dao = new YdbShardAssignmentsDao(schemeRoot, tableClient);
        return measure(dao, ShardAssignmentsDao.class);
    }

    @Deprecated
    @Bean
    ShardBalancerOptionsDao balancerOptionsDao() {
        var dao = new YdbShardBalancerOptionsDao(schemeRoot, tableClient);
        dao.migrateSchema();
        return measure(dao, ShardBalancerOptionsDao.class);
    }

    @Deprecated
    @Bean
    RootLink oldBalancerLink() {
        return new RootLink("/balancer-old", "Shard Balancer (OLD)");
    }

    @Bean
    RootLink balancerLink() {
        return new RootLink("/balancer", "Shard Balancer (NEW)");
    }

    @Bean
    GrpcBalancerService grpcBalancerService(LocalCoremonHost localHost, @NewBalancer DistributedLock lock) {
        return new GrpcBalancerService(localHost, lock, clock);
    }

    @Bean
    GrpcBalancerService.Proxy grpcBalancerServiceProxy(GrpcBalancerService service) {
        return service.createProxy();
    }

    private <TDao> TDao measure(TDao dao, Class<TDao> daoClass) {
        return DaoMetricsProxy.of(dao, daoClass, registry);
    }

    @Qualifier
    @Retention(RetentionPolicy.RUNTIME)
    @interface OldBalancer {
    }

    @Qualifier
    @Retention(RetentionPolicy.RUNTIME)
    @interface NewBalancer {
    }

    @ParametersAreNonnullByDefault
    private static class Bootstrap implements ApplicationListener<ContextRefreshedEvent> {
        //TODO: use BalancerHolderImpl here after migration <SOLOMON-6713>
        private final ShardBalancer balancer;

        Bootstrap(ShardBalancer balancer) {
            this.balancer = balancer;
        }

        @Override
        public void onApplicationEvent(ContextRefreshedEvent event) {
            balancer.start();
        }
    }

}
