package ru.yandex.solomon.coremon.balancer.cluster;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.IntSupplier;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.net.HostAndPort;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.discovery.DiscoveryServices;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.solomon.config.protobuf.coremon.TCoremonBalancerConfig;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.coremon.balancer.db.ShardAssignments;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Sergey Polovko
 */
public class CoremonCluster implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(CoremonCluster.class);

    // { fqdn -> host }
    private final ImmutableMap<String , CoremonHost> hosts;

    public CoremonCluster(
        String clientId,
        TCoremonBalancerConfig config,
        ThreadPoolProvider threadPoolProvider,
        LocalCoremonHost localNode,
        ClientOptionsFactory factory)
    {
        this.hosts = createHosts(clientId, config.getGrpcConfig(), threadPoolProvider, localNode, factory);
    }

    public void startPinging(long leaderSeqNo, IntSupplier totalAssignmentCount) {
        for (CoremonHost host : hosts.values()) {
            host.startPinging(leaderSeqNo, totalAssignmentCount);
        }
    }

    public void stopPinging() {
        for (CoremonHost host : hosts.values()) {
            host.stopPinging();
        }
    }

    private ImmutableMap<String, CoremonHost> createHosts(
        String clientId,
        TGrpcClientConfig config,
        ThreadPoolProvider threadPoolProvider,
        LocalCoremonHost localState,
        ClientOptionsFactory factory)
    {
        HostAndPort[] addresses = DiscoveryServices.resolve(config.getAddressesList())
            .stream()
            .distinct()
            .toArray(HostAndPort[]::new);

        ScheduledExecutorService timer = threadPoolProvider.getSchedulerExecutorService();
        var options = factory.newBuilder(
                "ShardBalancerConfig.GrpcConfig",
                config)
            .setClientId(clientId)
            .build();
        var hosts = ImmutableMap.<String, CoremonHost>builder();
        for (HostAndPort addr : addresses) {
            if (addr.getHost().equals(HostUtils.getFqdn()) || addr.getHost().equals("localhost")) {
                hosts.put(addr.getHost(), localState);
            } else {
                hosts.put(addr.getHost(), new RemoteCoremonHost(addr, options, timer));
            }
        }
        return hosts.build();
    }

    public Hosts getHosts(long offlineThresholdMillis) {
        List<CoremonHost> online = new ArrayList<>(hosts.size());
        List<CoremonHost> offline = new ArrayList<>(hosts.size());
        final long nowMillis = System.currentTimeMillis();

        for (CoremonHost host : hosts.values()) {
            if (nowMillis - host.getSeenAliveTimeMillis() < offlineThresholdMillis) {
                online.add(host);
            } else {
                offline.add(host);
            }
        }

        return new Hosts(online, offline);
    }

    public List<CoremonHost> getHosts() {
        return new ArrayList<>(hosts.values());
    }

    @Override
    public void close() throws Exception {
        for (CoremonHost host : hosts.values()) {
            host.close();
        }
    }

    @Nullable
    public CoremonHost getHost(String fqdn) {
        return hosts.get(fqdn);
    }

    public void updateAssignments(ShardAssignments assignments) {
        Map<String, IntSet> host2ShardIds = assignments.reverse();
        for (Map.Entry<String, IntSet> e : host2ShardIds.entrySet()) {
            CoremonHost host = hosts.get(e.getKey());
            if (host != null) {
                final String fqdn = host.getFqdn();
                host.setAssignments(e.getValue())
                    .whenComplete((aVoid, throwable) -> {
                        if (throwable != null) {
                            logger.warn("cannot sync assignments with {}", fqdn, throwable);
                        }
                    });
            } else {
                logger.warn("shards {} are assigned to unknown host {} ", e.getValue(), e.getKey());
            }
        }

        // unassign not assigned hosts
        for (var entry : hosts.entrySet()) {
            var fqdn = entry.getKey();
            var shardIds = host2ShardIds.get(fqdn);
            if (shardIds == null) {
                entry.getValue()
                    .setAssignments(new IntOpenHashSet(0))
                    .whenComplete((aVoid, throwable) -> {
                        if (throwable != null) {
                            logger.warn("cannot sync assignments with {}", fqdn, throwable);
                        }
                    });
            }
        }
    }

    /**
     * HOSTS
     */
    @Immutable
    public static class Hosts {

        private final ImmutableList<CoremonHost> online;
        private final ImmutableList<CoremonHost> offline;

        Hosts(List<CoremonHost> online, List<CoremonHost> offline) {
            this.online = ImmutableList.copyOf(online);
            this.offline = ImmutableList.copyOf(offline);
        }

        public ImmutableList<CoremonHost> getOnline() {
            return online;
        }

        public ImmutableList<CoremonHost> getOffline() {
            return offline;
        }
    }
}
