package ru.yandex.solomon.coremon.balancer.cluster;

import java.util.concurrent.CompletableFuture;
import java.util.function.IntSupplier;

import it.unimi.dsi.fastutil.ints.IntSet;

import ru.yandex.solomon.coremon.balancer.state.ShardIds;
import ru.yandex.solomon.coremon.balancer.state.ShardsLoadMap;

/**
 * @author Sergey Polovko
 */
public interface CoremonHost extends AutoCloseable {

    interface State {
        /**
         * @return set of shard ids assigned to host.
         */
        ShardIds getAssignments();

        /**
         * @return load information about shards currently processing by host.
         */
        ShardsLoadMap getShards();

        /**
         * @return true iff all assigned shard ids are equal to all shard ids in load map
         */
        boolean isSynced();

        long getUptimeMillis();

        long getCpuTimeNanos();

        long getMemoryBytes();

        long getNetworkBytes();
    }

    /**
     * @return fully qualified domain name of the host
     */
    String getFqdn();

    /**
     * @return wall clock time when the host was last seen available
     */
    long getSeenAliveTimeMillis();

    /**
     * Start sending pings to the host.
     *
     * @param leaderSeqNo current leader sequence number
     * @param totalShardCount total number of assigned shards
     */
    void startPinging(long leaderSeqNo, IntSupplier totalShardCount);

    /**
     * Stop pinging the host.
     */
    void stopPinging();

    /**
     * Get current host's state, including host load and last shards load.
     *
     * @return last state of the host
     */
    State getState(boolean refreshShardsStatus);

    /**
     * Assign particular shards to be served on the host.
     *
     * @param shardIds  set of shard ids
     * @return future for async operation
     */
    CompletableFuture<Void> setAssignments(IntSet shardIds);

    /**
     * Add or/and remove shards assigned to the host.
     *
     * @param shardIdsAdd     set of shard ids to be added
     * @param shardIdsRemove  set of shard ids to be removed
     * @return future for async operation
     */
    CompletableFuture<Void> changeAssignments(IntSet shardIdsAdd, IntSet shardIdsRemove);
}
