package ru.yandex.solomon.coremon.balancer.cluster;

import java.time.Clock;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import io.grpc.Status;
import io.grpc.stub.StreamObserver;
import it.unimi.dsi.fastutil.ints.IntSets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.InternalGrpcService;
import ru.yandex.monitoring.coremon.EShardState;
import ru.yandex.solomon.balancer.BalancerProto;
import ru.yandex.solomon.balancer.BalancerServiceGrpc;
import ru.yandex.solomon.balancer.TAssignShardRequest;
import ru.yandex.solomon.balancer.TAssignShardResponse;
import ru.yandex.solomon.balancer.TBalancerServiceGrpc;
import ru.yandex.solomon.balancer.TCreateShardRequest;
import ru.yandex.solomon.balancer.TCreateShardResponse;
import ru.yandex.solomon.balancer.TNodeSummary;
import ru.yandex.solomon.balancer.TPingRequest;
import ru.yandex.solomon.balancer.TPingResponse;
import ru.yandex.solomon.balancer.TShardSummary;
import ru.yandex.solomon.balancer.TShardSummary.EStatus;
import ru.yandex.solomon.balancer.TUnassignShardRequest;
import ru.yandex.solomon.balancer.TUnassignShardResponse;
import ru.yandex.solomon.balancer.protobuf.TBalancerResources;
import ru.yandex.solomon.coremon.balancer.state.ShardLoad;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.selfmon.ng.ProcSelfMon;
import ru.yandex.solomon.util.host.HostUtils;

import static java.util.Objects.requireNonNull;
import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcBalancerService extends TBalancerServiceGrpc.TBalancerServiceImplBase implements InternalGrpcService {
    private static final Logger logger = LoggerFactory.getLogger(GrpcBalancerService.class);

    private final Clock clock;
    private final LocalCoremonHost state;
    private final DistributedLock leader;
    private final long startedAt;

    public GrpcBalancerService(LocalCoremonHost state, DistributedLock leader, Clock clock) {
        this.state = state;
        this.leader = leader;
        this.clock = clock;
        this.startedAt = clock.millis();
    }

    @Override
    public void assignShard(TAssignShardRequest request, StreamObserver<TAssignShardResponse> responseObserver) {
        var future = ensureLeaderOwnership(request.getAssignmentSeqNo().getLeaderSeqNo())
                .thenCompose(node -> {
                    ensureDeadlineNotExpired(request.getExpiredAt());
                    var assignment = requireNonNull(BalancerProto.fromProto(request.getAssignmentSeqNo()));
                    logger.info("Receive shard assign {} {} from {}", request.getShardId(), assignment, node);

                    int numId = Integer.parseUnsignedInt(request.getShardId());
                    return state.changeAssignments(IntSets.singleton(numId), IntSets.EMPTY_SET);
                })
                .thenApply(ignore -> TAssignShardResponse.getDefaultInstance());
        asyncComplete(future, responseObserver);
    }

    @Override
    public void unassignShard(TUnassignShardRequest request, StreamObserver<TUnassignShardResponse> responseObserver) {
        var future = ensureLeaderOwnership(request.getAssignmentSeqNo().getLeaderSeqNo())
                .thenCompose(node -> {
                    ensureDeadlineNotExpired(request.getExpiredAt());
                    var assignment = requireNonNull(BalancerProto.fromProto(request.getAssignmentSeqNo()));
                    logger.info("Receive shard unassign {} {} from {}", request.getShardId(), assignment, node);

                    int numId = Integer.parseUnsignedInt(request.getShardId());
                    return state.changeAssignments(IntSets.EMPTY_SET, IntSets.singleton(numId));
                })
                .thenApply(ignore -> TUnassignShardResponse.getDefaultInstance());
        asyncComplete(future, responseObserver);
    }

    @Override
    public void ping(TPingRequest request, StreamObserver<TPingResponse> responseObserver) {
        var future = ensureLeaderOwnership(request.getLatestAssignmentSeqNo().getLeaderSeqNo())
            .thenApply(node -> {
                ensureDeadlineNotExpired(request.getExpiredAt());

                if (request.getTotalShardCountKnown()) {
                    state.setTotalShardCount(request.getTotalShardCount());
                }

                return TPingResponse.newBuilder()
                    .setNode(HostUtils.getFqdn())
                    .setNodeSummary(prepareNodeSummary())
                    .addAllShardSummary(prepareShardSummary())
                    .build();
            });
        asyncComplete(future, responseObserver);
    }

    private TNodeSummary prepareNodeSummary() {
        try {
            return TNodeSummary.newBuilder()
                    .setMemoryBytes(ProcSelfMon.getRssBytes())
                    .setUtimeMillis(ProcSelfMon.getUtimeMs())
                    .setUpTimeMillis(clock.millis() - startedAt)
                    .build();
        } catch (Throwable e) {
            // ProcSelfMon can not work on ci https://paste.yandex-team.ru/631268
            logger.error("prepare node summary failed: ", e);
            return TNodeSummary.getDefaultInstance();
        }
    }

    private List<TShardSummary> prepareShardSummary() {
        var shardsState = state.getState(true);
        var assignments = shardsState.getAssignments().getShards();
        var shardsLoad = shardsState.getShards();
        var result = new ArrayList<TShardSummary>(assignments.size());
        var it = assignments.iterator();
        while (it.hasNext()) {
            int numId = it.nextInt();
            var load = shardsLoad.get(numId);
            result.add(toShardSummary(numId, load));
        }
        return result;
    }

    private TShardSummary toShardSummary(int numId, @Nullable ShardLoad load) {
        if (load == null) {
            return TShardSummary.newBuilder()
                    .setShardId(Integer.toUnsignedString(numId))
                    .setStatus(EStatus.INIT)
                    .build();
        }

        return TShardSummary.newBuilder()
                .setShardId(Integer.toUnsignedString(numId))
                .setStatus(toStatus(load.getState()))
                .setUpTimeMillis(load.getUptimeMillis())
                .setResources(
                    TBalancerResources.newBuilder()
                        .setCpu(TimeUnit.NANOSECONDS.toMillis(load.getCpuTimeNanos()))
                        .setMetricsParseRate(load.getParsedMetrics())
                        .setMetricsCount(load.getMetricsCount())
                        .setNetworkBytes(load.getNetworkBytes())
                        .build())
                .build();
    }

    private EStatus toStatus(EShardState state) {
        switch (state) {
            case NEW:
            case UNKNOWN:
                return EStatus.INIT;
            case INDEXING:
            case LOADING:
                return EStatus.LOADING;
            default:
                return EStatus.READY;
        }
    }

    private CompletableFuture<String> ensureLeaderOwnership(long seqNo) {
        return leader.getLockDetail(seqNo)
                .thenApply(detail -> {
                    if (detail.isEmpty()) {
                        throw Status.ABORTED
                                .withDescription("Reject because leader ownership expired")
                                .asRuntimeException();
                    }

                    if (Long.compareUnsigned(seqNo, detail.get().seqNo()) != 0) {
                        throw Status.ABORTED
                                .withDescription("Rejected, seqNo mismatch("
                                        + seqNo
                                        + " != "
                                        + detail.get().seqNo()
                                        + "), leader now "
                                        + detail.get().owner())
                                .asRuntimeException();
                    }

                    return detail.get().owner();
                });
    }

    private void ensureDeadlineNotExpired(long expiredAt) {
        if (expiredAt == 0) {
            return;
        }

        if (System.currentTimeMillis() + 200L >= expiredAt) {
            throw Status.DEADLINE_EXCEEDED.asRuntimeException();
        }
    }

    public Proxy createProxy() {
        return new Proxy(this);
    }

    public static class Proxy extends BalancerServiceGrpc.BalancerServiceImplBase implements InternalGrpcService {
        private final GrpcBalancerService delegate;

        public Proxy(GrpcBalancerService delegate) {
            this.delegate = delegate;
        }

        @Override
        public void assignShard(TAssignShardRequest request, StreamObserver<TAssignShardResponse> responseObserver) {
            delegate.assignShard(request, responseObserver);
        }

        @Override
        public void unassignShard(TUnassignShardRequest request, StreamObserver<TUnassignShardResponse> responseObserver) {
            delegate.unassignShard(request, responseObserver);
        }

        @Override
        public void createShard(TCreateShardRequest request, StreamObserver<TCreateShardResponse> responseObserver) {
            delegate.createShard(request, responseObserver);
        }

        @Override
        public void ping(TPingRequest request, StreamObserver<TPingResponse> responseObserver) {
            delegate.ping(request, responseObserver);
        }
    }
}
