package ru.yandex.solomon.coremon.balancer.cluster;

import javax.annotation.Nullable;

import com.google.protobuf.Message;
import com.google.protobuf.TextFormat;
import io.grpc.Status;
import io.grpc.stub.StreamObserver;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.InternalGrpcService;
import ru.yandex.monitoring.coremon.CoremonBalancerServiceGrpc;
import ru.yandex.monitoring.coremon.TChangeAssignmentsRequest;
import ru.yandex.monitoring.coremon.TChangeAssignmentsResponse;
import ru.yandex.monitoring.coremon.THostLoad;
import ru.yandex.monitoring.coremon.TPingRequest;
import ru.yandex.monitoring.coremon.TPingResponse;

/**
 * @author Sergey Polovko
 */
public class RemoteCoremonHostPeer extends CoremonBalancerServiceGrpc.CoremonBalancerServiceImplBase
    implements InternalGrpcService
{
    private static final Logger logger = LoggerFactory.getLogger(RemoteCoremonHostPeer.class);

    private final LocalCoremonHost localNodeState;
    private volatile long lastSeenSeqNo = 0;

    public RemoteCoremonHostPeer(LocalCoremonHost localNodeState) {
        this.localNodeState = localNodeState;
    }

    @Override
    public void changeAssignments(TChangeAssignmentsRequest request, StreamObserver<TChangeAssignmentsResponse> observer) {
        if (!checkRequestExpired(request.getClass(), request.getExpiredAt(), observer)) {
            return;
        }

        if (!checkLeaderSeqNo(request.getLeaderSeqNo(), observer)) {
            return;
        }

        if (request.getShardIdsSetCount() != 0 &&
            (request.getShardIdsAddCount() != 0 || request.getShardIdsRemoveCount() != 0))
        {
            sendError(Status.FAILED_PRECONDITION, "set shardIds was provided with add/remove shardIds", null, observer);
            return;
        }

        try {
            if (request.getShardIdsAddCount() != 0 || request.getShardIdsRemoveCount() != 0) {
                var shardIdsAdd = new IntOpenHashSet(request.getShardIdsAddList());
                var shardIdsRemove = new IntOpenHashSet(request.getShardIdsRemoveList());
                localNodeState.changeAssignments(shardIdsAdd, shardIdsRemove);
            } else {
                // When Add and Remove sets both are empty, then this request must change
                // whole shard assignments, event if it is an empty set. Without it node
                // will never be initialized and will always respond 'Shard list not initialized yet'
                localNodeState.setAssignments(new IntOpenHashSet(request.getShardIdsSetList()));
            }
        } catch (Throwable t) {
            String msg = "cannot change shards assignments " +
                "{setCount=" + request.getShardIdsSetCount() +
                ", addCount=" + request.getShardIdsAddCount() +
                ", removeCount=" + request.getShardIdsRemoveCount() + '}';
            sendError(Status.INTERNAL, msg, t, observer);
            return;
        }

        sendResponse(TChangeAssignmentsResponse.getDefaultInstance(), observer);
    }

    @Override
    public void ping(TPingRequest request, StreamObserver<TPingResponse> observer) {
        if (!checkRequestExpired(request.getClass(), request.getExpiredAt(), observer)) {
            return;
        }

        if (!checkLeaderSeqNo(request.getLeaderSeqNo(), observer)) {
            return;
        }

        if (request.getTotalShardCountKnown()) {
            localNodeState.setTotalShardCount(request.getTotalShardCount());
        }

        try {
            CoremonHost.State state = localNodeState.getState(true);

            var response = TPingResponse.newBuilder()
                .setHostLoad(THostLoad.newBuilder()
                    .setUptimeMillis(state.getUptimeMillis())
                    .setCpuTimeNanos(state.getCpuTimeNanos())
                    .setMemoryBytes(state.getMemoryBytes())
                    .setNetworkBytes(state.getNetworkBytes()))
                    .setShardsLoad(state.getShards().toPb());
            sendResponse(response.build(), observer);
        } catch (Throwable t) {
            String msg = "cannot process ping request {" + TextFormat.shortDebugString(request) + '}';
            sendError(Status.INTERNAL, msg, t, observer);
        }
    }

    private static boolean checkRequestExpired(Class<?> requestType, long expiredAt, StreamObserver<?> observer) {
        if (expiredAt > 0 && System.currentTimeMillis() + 100L >= expiredAt) {
            String msg = "request " + requestType.getSimpleName() + "already expired (expiredAt=" + expiredAt + "ms)";
            sendError(Status.DEADLINE_EXCEEDED, msg, null, observer);
            return false;
        }
        return true;
    }

    private boolean checkLeaderSeqNo(long leaderSeqNo, StreamObserver<?> observer) {
        final long lastSeenSeqNo = this.lastSeenSeqNo;
        if (leaderSeqNo < lastSeenSeqNo) {
            String msg = "get request from impostor" +
                " (leaderSeqNo=" + leaderSeqNo +
                ", lastSeenSeqNo=" + lastSeenSeqNo + ')';
            sendError(Status.FAILED_PRECONDITION, msg, null, observer);
            return false;
        }
        if (leaderSeqNo > lastSeenSeqNo) {
            this.lastSeenSeqNo = leaderSeqNo;
        }
        return true;
    }

    private static <M extends Message> void sendResponse(M response, StreamObserver<M> observer) {
        observer.onNext(response);
        observer.onCompleted();
    }

    private static void sendError(
        Status status, String description, @Nullable Throwable cause, StreamObserver<?> observer)
    {
        logger.error(description, cause);
        observer.onError(status.withDescription(description)
            .withCause(cause)
            .asRuntimeException());
    }
}
