package ru.yandex.solomon.coremon.balancer.db;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.IntSet;

import static java.util.concurrent.CompletableFuture.delayedExecutor;
import static java.util.concurrent.CompletableFuture.runAsync;
import static java.util.concurrent.CompletableFuture.supplyAsync;

/**
 * @author Sergey Polovko
 */
public class MemShardAssignmentsDao implements ShardAssignmentsDao {

    private final Object lock = new Object();
    private final Int2ObjectMap<String> shard2Host = new Int2ObjectOpenHashMap<>();

    @Override
    public CompletableFuture<ShardAssignments> load() {
        return supplyAsync(() -> {
            synchronized (lock) {
                return ShardAssignments.copyOf(shard2Host);
            }
        }, delayedExecutor(10, TimeUnit.MILLISECONDS));
    }

    @Override
    public CompletableFuture<Void> save(ShardAssignments shard2Host) {
        return runAsync(() -> {
            synchronized (lock) {
                this.shard2Host.putAll(shard2Host.asMap());
            }
        }, delayedExecutor(10, TimeUnit.MILLISECONDS));
    }

    @Override
    public CompletableFuture<Void> delete(IntSet shardIds) {
        return runAsync(() -> {
            synchronized (lock) {
                for (var it = shardIds.iterator(); it.hasNext(); ) {
                    this.shard2Host.remove(it.nextInt());
                }
            }
        }, delayedExecutor(10, TimeUnit.MILLISECONDS));
    }

    @Override
    public CompletableFuture<Void> update(int shardId, String host) {
        return runAsync(() -> {
            synchronized (lock) {
                this.shard2Host.computeIfPresent(shardId, (k, v) -> host);
            }
        }, delayedExecutor(10, TimeUnit.MILLISECONDS));
    }
}
