package ru.yandex.solomon.coremon.balancer.db;

import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;

import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMaps;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import it.unimi.dsi.fastutil.ints.IntSets;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
@Immutable
public class ShardAssignments {

    public static final ShardAssignments EMPTY = new ShardAssignments(Int2ObjectMaps.emptyMap());

    private final Int2ObjectMap<String> shard2Host;

    private ShardAssignments(Int2ObjectMap<String> shard2Host) {
        this.shard2Host = Objects.requireNonNull(shard2Host, "shard2Host");
    }

    public static ShardAssignments copyOf(Map<Integer, String> shard2Host) {
        if (shard2Host.isEmpty()) {
            return EMPTY;
        }
        return new ShardAssignments(new Int2ObjectOpenHashMap<>(shard2Host));
    }

    public static ShardAssignments ownOf(Int2ObjectMap<String> shard2Host) {
        if (shard2Host.isEmpty()) {
            return EMPTY;
        }
        return new ShardAssignments(shard2Host);
    }

    public static ShardAssignments ofShardIds(String host, IntSet shardIds) {
        if (shardIds.isEmpty()) {
            return EMPTY;
        }
        var newAssignments = new Int2ObjectOpenHashMap<String>(shardIds.size());
        for (var it = shardIds.iterator(); it.hasNext(); ) {
            newAssignments.put(it.nextInt(), host);
        }
        return new ShardAssignments(newAssignments);
    }

    public static ShardAssignments combine(List<ShardAssignments> assignments) {
        if (assignments.isEmpty()) {
            return EMPTY;
        }
        int totalSize = assignments.stream().mapToInt(ShardAssignments::size).sum();
        var map = new Int2ObjectOpenHashMap<String>(totalSize);
        for (ShardAssignments a : assignments) {
            map.putAll(a.shard2Host);
        }
        return new ShardAssignments(map);
    }

    public static ShardAssignments singleton(int shardId, String host) {
        return new ShardAssignments(Int2ObjectMaps.singleton(shardId, host));
    }

    public Map<String, IntSet> reverse() {
        if (isEmpty()) {
            return Map.of();
        }
        var host2Shards = new Object2ObjectOpenHashMap<String, IntSet>();
        for (Int2ObjectMap.Entry<String> e : shard2Host.int2ObjectEntrySet()) {
            int shardId = e.getIntKey();
            String host = e.getValue();
            IntSet shardIds = host2Shards.get(host);
            if (shardIds == null) {
                shardIds = new IntOpenHashSet();
                host2Shards.put(host, shardIds);
            }
            shardIds.add(shardId);
        }
        return host2Shards;
    }

    @Nullable
    public String get(int shardId) {
        return shard2Host.get(shardId);
    }

    public int size() {
        return shard2Host.size();
    }

    public boolean isEmpty() {
        return shard2Host.isEmpty();
    }

    public IntSet getIds() {
        return IntSets.unmodifiable(shard2Host.keySet());
    }

    public ShardAssignments mergeWith(ShardAssignments rhs) {
        if (this == rhs || rhs.isEmpty()) {
            return this;
        }
        if (isEmpty()) {
            return rhs;
        }
        var map = new Int2ObjectOpenHashMap<String>(size() + rhs.size());
        map.putAll(shard2Host);
        map.putAll(rhs.shard2Host);
        return new ShardAssignments(map);
    }

    public ShardAssignments delete(IntSet toDelete) {
        if (toDelete.isEmpty()) {
            return this;
        }
        var map = new Int2ObjectOpenHashMap<String>(size());
        for (var e : shard2Host.int2ObjectEntrySet()) {
            if (!toDelete.contains(e.getIntKey())) {
                map.put(e.getIntKey(), e.getValue());
            }
        }
        return map.size() == size() ? this : new ShardAssignments(map);
    }

    public Int2ObjectMap<String> asMap() {
        return Int2ObjectMaps.unmodifiable(shard2Host);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ShardAssignments that = (ShardAssignments) o;
        return shard2Host.equals(that.shard2Host);
    }

    @Override
    public int hashCode() {
        return shard2Host.hashCode();
    }

    @Override
    public String toString() {
        var sb = new StringBuilder("ShardAssignments{");
        var it = shard2Host.int2ObjectEntrySet().iterator();
        for (int i = 0; i < 5 && it.hasNext(); i++) {
            if (i > 0) {
                sb.append(", ");
            }
            Int2ObjectMap.Entry<String> e = it.next();
            sb.append(Integer.toUnsignedLong(e.getIntKey()));
            sb.append(": ").append(e.getValue());
        }
        if (it.hasNext()) {
            sb.append(", ... (").append(shard2Host.size() - 5).append(" more)");
        }
        sb.append("}");
        return sb.toString();
    }
}
