package ru.yandex.solomon.coremon.balancer.db;

import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.google.common.collect.ImmutableSet;

/**
 * @author Sergey Polovko
 */
public class ShardBalancerOptions {

    public static final ShardBalancerOptions DEFAULT = new ShardBalancerOptions(
        TimeUnit.MINUTES.toMillis(2),
        1,
        1.0,
        1.0, 1.0, 1.0,
        ImmutableSet.of(),
        false);

    private final long offlineThresholdMillis;
    private final int rebalaceShardsInFlight;
    private final double rebalaceThreshold;
    private final double cpuWeightFactor;
    private final double memoryWeightFactor;
    private final double networkWeightFactor;
    private final ImmutableSet<String> inactiveHosts;
    private final boolean useNewBalancer;

    public ShardBalancerOptions(
        long offlineThresholdMillis,
        int rebalaceShardsInFlight,
        double rebalaceThreshold,
        double cpuWeightFactor,
        double memoryWeightFactor,
        double networkWeightFactor,
        ImmutableSet<String> inactiveHosts,
        boolean useNewBalancer)
    {
        this.offlineThresholdMillis = offlineThresholdMillis;
        this.rebalaceShardsInFlight = rebalaceShardsInFlight;
        this.rebalaceThreshold = rebalaceThreshold;
        this.cpuWeightFactor = cpuWeightFactor;
        this.memoryWeightFactor = memoryWeightFactor;
        this.networkWeightFactor = networkWeightFactor;
        this.inactiveHosts = inactiveHosts;
        this.useNewBalancer = useNewBalancer;
    }

    public long getOfflineThresholdMillis() {
        return offlineThresholdMillis;
    }

    public int getRebalaceShardsInFlight() {
        return rebalaceShardsInFlight;
    }

    public double getRebalaceThreshold() {
        return rebalaceThreshold;
    }

    public double getCpuWeightFactor() {
        return cpuWeightFactor;
    }

    public double getMemoryWeightFactor() {
        return memoryWeightFactor;
    }

    public double getNetworkWeightFactor() {
        return networkWeightFactor;
    }

    public ImmutableSet<String> getInactiveHosts() {
        return inactiveHosts;
    }

    public boolean isUseNewBalancer() {
        return useNewBalancer;
    }

    public ShardBalancerOptions withInactiveHosts(Set<String> hosts) {
        return new ShardBalancerOptions(
            offlineThresholdMillis,
            rebalaceShardsInFlight,
            rebalaceThreshold,
            cpuWeightFactor,
            memoryWeightFactor,
            networkWeightFactor,
            ImmutableSet.copyOf(hosts),
            useNewBalancer);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ShardBalancerOptions that = (ShardBalancerOptions) o;

        if (offlineThresholdMillis != that.offlineThresholdMillis) return false;
        if (rebalaceShardsInFlight != that.rebalaceShardsInFlight) return false;
        if (Double.compare(that.rebalaceThreshold, rebalaceThreshold) != 0) return false;
        if (Double.compare(that.cpuWeightFactor, cpuWeightFactor) != 0) return false;
        if (Double.compare(that.memoryWeightFactor, memoryWeightFactor) != 0) return false;
        if (Double.compare(that.networkWeightFactor, networkWeightFactor) != 0) return false;
        if (useNewBalancer != that.useNewBalancer) return false;
        return inactiveHosts.equals(that.inactiveHosts);
    }

    @Override
    public int hashCode() {
        int result;
        long temp;
        result = (int) (offlineThresholdMillis ^ (offlineThresholdMillis >>> 32));
        result = 31 * result + rebalaceShardsInFlight;
        temp = Double.doubleToLongBits(rebalaceThreshold);
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        temp = Double.doubleToLongBits(cpuWeightFactor);
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        temp = Double.doubleToLongBits(memoryWeightFactor);
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        temp = Double.doubleToLongBits(networkWeightFactor);
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        result = 31 * result + inactiveHosts.hashCode();
        result = 31 * result + (useNewBalancer ? 1 : 0);
        return result;
    }

    @Override
    public String toString() {
        return "ShardBalancerOptions{" +
            "offlineThresholdMillis=" + offlineThresholdMillis +
            ", rebalaceShardsInFlight=" + rebalaceShardsInFlight +
            ", rebalaceThreshold=" + rebalaceThreshold +
            ", cpuWeightFactor=" + cpuWeightFactor +
            ", memoryWeightFactor=" + memoryWeightFactor +
            ", networkWeightFactor=" + networkWeightFactor +
            ", inactiveHosts=" + inactiveHosts +
            ", useNewBalancer=" + useNewBalancer +
            '}';
    }
}
