package ru.yandex.solomon.coremon.balancer.db;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableSet;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.AlterTableSettings;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.transaction.TxControl;
import com.yandex.ydb.table.values.PrimitiveType;
import org.apache.commons.lang3.StringUtils;

import static com.yandex.ydb.table.values.PrimitiveValue.bool;
import static com.yandex.ydb.table.values.PrimitiveValue.float64;
import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.uint64;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Sergey Polovko
 */
public class YdbShardBalancerOptionsDao implements ShardBalancerOptionsDao {

    private static final char HOSTS_DELIMITER = ';';

    private final String tablePath;
    private final SessionRetryContext sessionCtx;
    private final String saveQuery;
    private final String loadQuery;

    public YdbShardBalancerOptionsDao(String rootPath, TableClient tableClient) {
        this.tablePath = rootPath + "/ShardBalancer";
        this.sessionCtx = SessionRetryContext.create(tableClient)
            .maxRetries(10)
            .executor(ForkJoinPool.commonPool())
            .build();

        this.saveQuery = String.format("""
                --!syntax_v1
                DECLARE $offlineThresholdMillis AS Uint64;
                DECLARE $rebalaceShardsInFlight AS Int32;
                DECLARE $rebalaceThreshold AS Double;
                DECLARE $cpuWeightFactor AS Double;
                DECLARE $memoryWeightFactor AS Double;
                DECLARE $networkWeightFactor AS Double;
                DECLARE $inactiveHosts AS Utf8;
                DECLARE $useNewBalancer AS Bool;
                REPLACE INTO `%s`(
                  id,
                  offlineThresholdMillis,
                  rebalaceShardsInFlight,
                  rebalaceThreshold,
                  cpuWeightFactor,
                  memoryWeightFactor,
                  networkWeightFactor,
                  inactiveHosts,
                  useNewBalancer)
                VALUES (
                  1,
                  $offlineThresholdMillis,
                  $rebalaceShardsInFlight,
                  $rebalaceThreshold,
                  $cpuWeightFactor,
                  $memoryWeightFactor,
                  $networkWeightFactor,
                  $inactiveHosts,
                  $useNewBalancer);
                """, tablePath);

        this.loadQuery = String.format("""
                SELECT * FROM `%s` WHERE id = 1;
                """, tablePath);
    }

    public CompletableFuture<Void> migrateSchema() {
        return sessionCtx.supplyResult(session -> session.describeTable(tablePath))
            .thenCompose(result -> {
                if (!result.isSuccess()) {
                    return createSchemaForTests();
                }

                return alterTable(result.expect("unable describe " + tablePath));
            });
    }

    private CompletableFuture<Void> alterTable(TableDescription table) {
        for (var column : table.getColumns()) {
            if ("useNewBalancer".equals(column.getName())) {
                return completedFuture(null);
            }
        }

        var settings = new AlterTableSettings();
        settings.addColumn("useNewBalancer", PrimitiveType.bool().makeOptional());
        return sessionCtx.supplyStatus(session -> session.alterTable(tablePath, settings))
            .thenAccept(status -> status.expect("unable alter " + tablePath));
    }

    @Override
    public CompletableFuture<ShardBalancerOptions> load() {
        return execute(loadQuery, Params.empty())
            .thenApply(result -> {
                DataQueryResult queryResult = result.expect("cannot load balancer options");
                ResultSetReader resultSet = queryResult.getResultSet(0);
                if (!resultSet.next()) {
                    return ShardBalancerOptions.DEFAULT;
                }

                var hosts = StringUtils.split(resultSet.getColumn("inactiveHosts").getUtf8(), HOSTS_DELIMITER);
                return new ShardBalancerOptions(
                    resultSet.getColumn("offlineThresholdMillis").getUint64(),
                    resultSet.getColumn("rebalaceShardsInFlight").getInt32(),
                    resultSet.getColumn("rebalaceThreshold").getFloat64(),
                    resultSet.getColumn("cpuWeightFactor").getFloat64(),
                    resultSet.getColumn("memoryWeightFactor").getFloat64(),
                    resultSet.getColumn("networkWeightFactor").getFloat64(),
                    ImmutableSet.copyOf(hosts),
                    resultSet.getColumn("useNewBalancer").getBool());
            });
    }

    @Override
    public CompletableFuture<Void> save(ShardBalancerOptions options) {
        Params params = Params.create()
            .put("$offlineThresholdMillis", uint64(options.getOfflineThresholdMillis()))
            .put("$rebalaceShardsInFlight",  int32(options.getRebalaceShardsInFlight()))
            .put("$rebalaceThreshold",  float64(options.getRebalaceThreshold()))
            .put("$cpuWeightFactor", float64(options.getCpuWeightFactor()))
            .put("$memoryWeightFactor", float64(options.getMemoryWeightFactor()))
            .put("$networkWeightFactor", float64(options.getNetworkWeightFactor()))
            .put("$inactiveHosts", utf8(StringUtils.join(options.getInactiveHosts(), HOSTS_DELIMITER)))
            .put("$useNewBalancer", bool(options.isUseNewBalancer()));

        return execute(saveQuery, params)
            .thenAccept(result -> result.expect("cannot save balancer options"));
    }

    @VisibleForTesting
    public CompletableFuture<Void> createSchemaForTests() {
        var tableDesc = TableDescription.newBuilder()
            .addNullableColumn("id", PrimitiveType.uint32())
            .addNullableColumn("offlineThresholdMillis", PrimitiveType.uint64())
            .addNullableColumn("rebalaceShardsInFlight", PrimitiveType.int32())
            .addNullableColumn("rebalaceThreshold", PrimitiveType.float64())
            .addNullableColumn("cpuWeightFactor", PrimitiveType.float64())
            .addNullableColumn("memoryWeightFactor", PrimitiveType.float64())
            .addNullableColumn("networkWeightFactor", PrimitiveType.float64())
            .addNullableColumn("inactiveHosts", PrimitiveType.utf8())
            .addNullableColumn("useNewBalancer", PrimitiveType.bool())
            .setPrimaryKey("id")
            .build();

        return sessionCtx.supplyStatus(s -> s.createTable(tablePath, tableDesc))
            .thenAccept(status -> status.expect("cannot create table " + tablePath));
    }

    @VisibleForTesting
    CompletableFuture<Void> dropSchemaForTests() {
        return sessionCtx.supplyStatus(s -> s.dropTable(tablePath))
            .thenAccept(status -> status.expect("cannot drop table " + tablePath));
    }

    private CompletableFuture<Result<DataQueryResult>> execute(String query, Params params) {
        try {
            return sessionCtx.supplyResult(s -> {
                var settings = new ExecuteDataQuerySettings().keepInQueryCache();
                var tx = TxControl.serializableRw();
                return s.executeDataQuery(query, tx, params, settings);
            });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }
}
