package ru.yandex.solomon.coremon.balancer.state;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import com.google.common.base.Preconditions;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectMaps;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.ints.IntSet;
import it.unimi.dsi.fastutil.ints.IntSets;
import it.unimi.dsi.fastutil.longs.LongArrayList;

import ru.yandex.monitoring.coremon.EShardState;
import ru.yandex.monitoring.coremon.TShardsLoad;

/**
 * @author Sergey Polovko
 */
@Immutable
public final class ShardsLoadMap {

    public static final ShardsLoadMap EMPTY = new ShardsLoadMap(Int2ObjectMaps.emptyMap());

    private final Int2ObjectMap<ShardLoad> shards;
    private final long idsHash;

    private ShardsLoadMap(Int2ObjectMap<ShardLoad> shards) {
        this.shards = shards;
        this.idsHash = ShardIds.ofWholeShards(shards.keySet()).getHash();
    }

    public static ShardsLoadMap ownOf(Int2ObjectMap<ShardLoad> shards) {
        return shards.isEmpty() ? EMPTY : new ShardsLoadMap(shards);
    }

    public static ShardsLoadMap copyOf(Map<Integer, ShardLoad> shards) {
        return shards.isEmpty() ? EMPTY : new ShardsLoadMap(new Int2ObjectOpenHashMap<>(shards));
    }

    public IntSet getIds() {
        return IntSets.unmodifiable(shards.keySet());
    }

    public int size() {
        return shards.size();
    }

    public static ShardsLoadMap combine(ShardsLoadMap... maps) {
        int totalSize = Arrays.stream(maps).mapToInt(ShardsLoadMap::size).sum();
        var shards = new Int2ObjectOpenHashMap<ShardLoad>(totalSize);
        for (ShardsLoadMap map : maps) {
            shards.putAll(map.shards);
        }
        return ownOf(shards);
    }

    public long getIdsHash() {
        return idsHash;
    }

    public ShardsLoadMap retainAll(IntSet ids) {
        if (shards.isEmpty()) {
            return this;
        }
        if (ids.isEmpty()) {
            return EMPTY;
        }

        var shardsCopy = new Int2ObjectOpenHashMap<>(shards);
        shardsCopy.keySet().retainAll(ids);
        return shardsCopy.isEmpty() ? EMPTY : new ShardsLoadMap(shardsCopy);
    }

    @Nullable
    public ShardLoad get(int shardId) {
        return shards.get(shardId);
    }

    public Collection<ShardLoad> values() {
        return Collections.unmodifiableCollection(shards.values());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        ShardsLoadMap s = (ShardsLoadMap) o;
        if (idsHash != s.idsHash) return false;
        return shards.equals(s.shards);
    }

    @Override
    public int hashCode() {
        int result = shards.hashCode();
        result = 31 * result + (int) (idsHash ^ (idsHash >>> 32));
        return result;
    }

    @Override
    public String toString() {
        var sb = new StringBuilder();
        sb.append("ShardsMap{idsHash=").append(Long.toHexString(idsHash));
        sb.append(", shards=[");

        // output maximum 5 entries
        var it = shards.int2ObjectEntrySet().iterator();
        for (int i = 0; it.hasNext() && i < 5; i++) {
            if (i > 0) {
                sb.append(", ");
            }
            Int2ObjectMap.Entry<ShardLoad> e = it.next();
            sb.append(Integer.toUnsignedLong(e.getIntKey())).append('=');
            e.getValue().toString(sb);
        }
        if (it.hasNext()) {
            sb.append(", ... (").append(shards.size() - 5).append(" more)");
        }
        sb.append("]}");
        return sb.toString();
    }

    public static ShardsLoadMap fromPb(TShardsLoad pb) {
        final int count = pb.getShardIdsCount();
        final boolean sameSize =
            count == pb.getStatesCount() &&
            count == pb.getUptimeMillisCount() &&
            count == pb.getCpuTimeNanosCount() &&
            count == pb.getMetricsCountCount() &&
            count == pb.getNetworkBytesCount();

        Preconditions.checkArgument(
            sameSize,
            "inconsistent size of arrays: " +
            "ShardIds=%d, States=%d, UptimeMillis=%d, " +
            "CpuTimeNanos=%d, MetricsCount=%d, NetworkBytes=%d",
            pb.getShardIdsCount(),
            pb.getStatesCount(),
            pb.getUptimeMillisCount(),
            pb.getCpuTimeNanosCount(),
            pb.getMetricsCountCount(),
            pb.getNetworkBytesCount());

        if (count == 0) {
            return EMPTY;
        }

        final var map = new Int2ObjectOpenHashMap<ShardLoad>(count);
        for (int i = 0; i < count; i++) {
            final int shardId = pb.getShardIds(i);
            final EShardState state = pb.getStates(i);
            final long uptimeMillis = pb.getUptimeMillis(i);
            final long cpuTimeNanos = pb.getCpuTimeNanos(i);
            final long metricsCount = pb.getMetricsCount(i);
            final long networkBytes = pb.getNetworkBytes(i);
            map.put(shardId, new ShardLoad(shardId, state, uptimeMillis, cpuTimeNanos, metricsCount, networkBytes, 0));
        }

        return new ShardsLoadMap(map);
    }

    public TShardsLoad toPb() {
        final int size = shards.size();
        if (size == 0) {
            return TShardsLoad.getDefaultInstance();
        }

        // put all data in local arrays first to avoid reallocations and
        // redundant checks inside protobuf message builder

        final var shardIds = new IntArrayList(size);
        final var states = new ArrayList<EShardState>(size);
        final var uptimeMillis = new LongArrayList(size);
        final var cpuTimeNanos = new LongArrayList(size);
        final var metricsCount = new LongArrayList(size);
        final var networkBytes = new LongArrayList(size);

        for (ShardLoad s : shards.values()) {
            shardIds.add(s.getId());
            states.add(s.getState());
            uptimeMillis.add(s.getUptimeMillis());
            cpuTimeNanos.add(s.getCpuTimeNanos());
            metricsCount.add(s.getMetricsCount());
            networkBytes.add(s.getNetworkBytes());
        }

        return TShardsLoad.newBuilder()
            .addAllShardIds(shardIds)
            .addAllStates(states)
            .addAllUptimeMillis(uptimeMillis)
            .addAllCpuTimeNanos(cpuTimeNanos)
            .addAllMetricsCount(metricsCount)
            .addAllNetworkBytes(networkBytes)
            .build();
    }
}
