package ru.yandex.solomon.coremon.balancer.www;

import java.time.Duration;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.coremon.balancer.ShardBalancer;
import ru.yandex.solomon.coremon.balancer.cluster.CoremonHost;
import ru.yandex.solomon.coremon.balancer.db.ShardBalancerOptions;
import ru.yandex.solomon.coremon.balancer.db.ShardBalancerOptionsDao;
import ru.yandex.solomon.staffOnly.RedirectException;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Sergey Polovko
 */
@RestController
public class ShardBalancerController {
    private final HttpAuthenticator authenticator;
    private final InternalAuthorizer authorizer;
    private final ShardBalancer balancer;
    private final ShardBalancerOptionsDao balancerOptionsDao;
    private final int port;

    public ShardBalancerController(
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer,
            ShardBalancer balancer,
            ShardBalancerOptionsDao balancerOptionsDao,
            int port)
    {
        this.authenticator = authenticator;
        this.authorizer = authorizer;
        this.balancer = balancer;
        this.balancerOptionsDao = balancerOptionsDao;
        this.port = port;
    }

    @RequestMapping(value = "/balancer-old", produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> balancer(ServerHttpRequest request) {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenCompose(account -> balancerImpl(request));
    }

    private CompletableFuture<String> balancerImpl(ServerHttpRequest request) {
        redirectToLeaderIfNeeded(request);

        return balancerOptionsDao.load()
            .thenApply(options -> {
                int sortBy = Integer.parseInt(Nullables.orDefault(request.getQueryParams().getFirst("sortBy"), "1"));
                return new ShardBalancerPage(balancer, options, sortBy, port).genString();
            });
    }

    @RequestMapping("/balancer-old/changeHostState")
    public CompletableFuture<?> changeHostState(
        @RequestParam(value = "fqdn") String fqdn,
        @RequestParam(value = "active") boolean active,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenCompose(account -> changeHostStateImpl(fqdn, active, request));
    }

    private CompletableFuture<?> changeHostStateImpl(String fqdn, boolean active, ServerHttpRequest request) {
        redirectToLeaderIfNeeded(request);

        return balancerOptionsDao.load()
            .thenCompose(options -> {
                var hosts = new HashSet<>(options.getInactiveHosts());
                if ("ALL".equals(fqdn)) {
                    List<String> allHosts = balancer.getCluster().getHosts().stream()
                        .map(CoremonHost::getFqdn)
                        .collect(Collectors.toList());
                    if (active) {
                        hosts.removeAll(allHosts);
                    } else {
                        hosts.addAll(allHosts);
                    }
                } else {
                    if (active) {
                        hosts.remove(fqdn);
                    } else {
                        hosts.add(fqdn);
                    }
                }
                return balancerOptionsDao.save(options.withInactiveHosts(hosts));
            })
            .thenAccept(aVoid -> redirectBack(request));
    }

    @RequestMapping("/balancer-old/saveOptions")
    public CompletableFuture<?> saveOptions(ServerHttpRequest request) {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenCompose(account -> saveOptionsImpl(request));
    }

    private CompletableFuture<?> saveOptionsImpl(ServerHttpRequest request) {
        redirectToLeaderIfNeeded(request);

        return balancerOptionsDao.load()
            .thenCompose(prevOpts -> {
                var opts = new ShardBalancerOptions(
                    getTime(request, "offlineThresholdMillis"),
                    getInt(request, "rebalaceShardsInFlight"),
                    getDouble(request, "rebalaceThreshold"),
                    getDouble(request, "cpuWeightFactor"),
                    getDouble(request, "memoryWeightFactor"),
                    getDouble(request, "networkWeightFactor"),
                    prevOpts.getInactiveHosts(),
                    getBoolean(request, "useNewBalancer"));

                return balancerOptionsDao.save(opts);
            })
            .thenAccept(aVoid -> redirectBack(request));
    }

    @RequestMapping("/balancer-old/moveShard")
    public CompletableFuture<?> moveShard(
        @RequestParam(value = "fqdn") String fqdn,
        @RequestParam(value = "shardNumId", required = false) Integer numId,
        @RequestParam(value = "shardStrId", required = false) String strId,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenCompose(account -> moveShardImpl(fqdn, numId, strId, request));
    }

    private CompletableFuture<?> moveShardImpl(String fqdn, Integer numId, String strId, ServerHttpRequest request) {
        redirectToLeaderIfNeeded(request);

        if (numId == null && strId == null) {
            redirectBack(request);
            return CompletableFuture.completedFuture(null);
        }

        return balancer.moveShard(fqdn, numId, strId)
            .thenAcceptAsync(aVoid -> redirectBack(request));
    }

    @RequestMapping("/balancer-old/kickShards")
    public CompletableFuture<Void> kickShards(
        @RequestParam(value = "fqdn") String fqdn,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenCompose(account -> kickShardsImpl(fqdn, request));
    }

    @RequestMapping("/balancer-old/kickShard")
    public CompletableFuture<Void> kickShard(
            @RequestParam(value = "shardId") String shardId,
            @RequestParam(value = "allowNewBalancer", defaultValue = "false") boolean allowNewBalancer,
            ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
                .thenCompose(authorizer::authorize)
                .thenCompose(account -> kickShardImpl(shardId, allowNewBalancer, request));
    }

    private CompletableFuture<Void> kickShardImpl(String shardId, boolean allowNewBalancer, ServerHttpRequest request) {
        redirectToLeaderIfNeeded(request);
        return balancer.kickShard(shardId, allowNewBalancer)
                .thenAcceptAsync(aVoid -> redirectBack(request));
    }

    private CompletableFuture<Void> kickShardsImpl(String fqdn, ServerHttpRequest request) {
        redirectToLeaderIfNeeded(request);
        return balancer.kickShardsFrom(fqdn)
            .thenAcceptAsync(aVoid -> redirectBack(request));
    }

    private static long getTime(ServerHttpRequest request, String param) {
        return DurationUtils.parseDuration(request.getQueryParams().getFirst(param))
            .orElse(Duration.ZERO)
            .toMillis();
    }

    private static int getInt(ServerHttpRequest request, String param) {
        String value = request.getQueryParams().getFirst(param);
        if (value == null || "".equals(value)) {
            return 0;
        }
        return Integer.parseInt(value);
    }

    private static double getDouble(ServerHttpRequest request, String param) {
        String value = request.getQueryParams().getFirst(param);
        if (value == null || "".equals(value)) {
            return 0.0;
        }
        return Double.parseDouble(value);
    }

    private static boolean getBoolean(ServerHttpRequest request, String param) {
        String value = request.getQueryParams().getFirst(param);
        if (value == null || "".equals(value)) {
            return false;
        }
        return Boolean.parseBoolean(value);
    }

    private void redirectToLeaderIfNeeded(ServerHttpRequest request) {
        if (balancer.isLeaderOnThisHost()) {
            return;
        }

        Optional<String> leaderHost = balancer.getLeaderHost();
        if (leaderHost.isEmpty()) {
            throw new IllegalStateException("leader is unknown");
        }

        String url = "http://" + leaderHost.get() + ":" + port + targetUrl(request);
        throw new RedirectException(url);
    }
    private void redirectBack(ServerHttpRequest request) {
        String referer = request.getHeaders().getFirst("Referer");
        if (StringUtils.isNotEmpty(referer)) {
            throw new RedirectException(referer);
        }
        throw new RedirectException("/balancer-old");
    }

    private String targetUrl(ServerHttpRequest request) {
        var uri = request.getURI();
        return Strings.isNullOrEmpty(uri.getQuery())
                ? uri.getPath()
                : uri.getPath() + "?" + uri.getQuery();
    }

}
