package ru.yandex.solomon.coremon.balancer.www;

import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.IntConsumer;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.monitoring.coremon.EShardState;
import ru.yandex.solomon.coremon.balancer.ShardBalancer;
import ru.yandex.solomon.coremon.balancer.cluster.CoremonCluster;
import ru.yandex.solomon.coremon.balancer.cluster.CoremonHost;
import ru.yandex.solomon.coremon.balancer.db.ShardBalancerOptions;
import ru.yandex.solomon.coremon.balancer.state.LoadCalc;
import ru.yandex.solomon.coremon.balancer.state.ShardLoad;
import ru.yandex.solomon.staffOnly.html.HtmlWriter;
import ru.yandex.solomon.staffOnly.www.ManagerPageTemplate;
import ru.yandex.solomon.util.host.HostUtils;
import ru.yandex.solomon.util.time.DurationUtils;

import static ru.yandex.solomon.util.time.DurationUtils.formatDurationMillis;

/**
 * @author Sergey Polovko
 */
public class ShardBalancerPage extends ManagerPageTemplate {

    private final ShardBalancer balancer;
    private final ShardBalancerOptions options;
    private final int sortBy;
    private final int port;

    ShardBalancerPage(ShardBalancer balancer, ShardBalancerOptions options, int sortBy, int port) {
        super("Shard Balancer (OLD)");
        this.balancer = balancer;
        this.options = options;
        this.sortBy = sortBy;
        this.port = port;
    }

    @Override
    public void commonCssJs() {
        super.commonCssJs();
        tag("style", () -> {
            writeRaw(".table-hover tbody tr:hover > td { background-color: #9dddf2 }");
        });
        script(() -> {
            writeRaw("setTimeout(function () { location.reload(1); }, 30000);");
        });
    }

    @Override
    protected void content() {
        tag("div.row", () -> {
            tag("div.col-md-6", () -> {
                tag("div.well", () -> {
                    writeOptionsForm(options);
                });
            });
            tag("div.col-md-6", () -> {
                tag("div.well", this::writeShardMoveForm);
            });
        });
        writeHostsTable(balancer.getCluster(), options);
    }

    private void writeOptionsForm(ShardBalancerOptions opts) {
        ShardBalancerOptions defs = ShardBalancerOptions.DEFAULT;
        form(() -> {
            tag("div.form-group", () -> {
                h3("Timeouts");
                table(() -> {
                    timeInput(
                        "offlineThresholdMillis",
                        "Offline timeout:",
                        opts.getOfflineThresholdMillis(),
                        defs.getOfflineThresholdMillis());
                });
            });

            tag("div.form-group", () -> {
                h3("Rebalance");
                table(() -> {
                    intInput(
                        "rebalaceShardsInFlight",
                        "Max in-flight shards:",
                        opts.getRebalaceShardsInFlight(),
                        defs.getRebalaceShardsInFlight());

                    doubleInput(
                        "rebalaceThreshold",
                        "Rebalance weight threshold:",
                        opts.getRebalaceThreshold(),
                        defs.getRebalaceThreshold());

                    doubleInput(
                        "cpuWeightFactor",
                        "CPU weight factor:",
                        opts.getCpuWeightFactor(),
                        defs.getCpuWeightFactor());

                    doubleInput(
                        "memoryWeightFactor",
                        "Memory weight factor:",
                        opts.getMemoryWeightFactor(),
                        defs.getMemoryWeightFactor());

                    doubleInput(
                        "networkWeightFactor",
                        "Network weight factor:",
                        opts.getNetworkWeightFactor(),
                        defs.getNetworkWeightFactor());

                    booleanInput(
                        "useNewBalancer",
                        "Use new balancer:",
                        opts.isUseNewBalancer(),
                        defs.isUseNewBalancer());
                });
            });
            tag("div.form-group", () -> {
                buttonSubmitDefault("Save");
            });
        }, new Attr("action", "/balancer-old/saveOptions"), new Attr("class", "form-horizontal"));
    }

    private void writeShardMoveForm() {
        form(() -> {
            tag("div.form-group", () -> {
                h3("Move Shard (fill either num or string ID)");
                textInput("shardNumId", "Shard Num ID:", "", "(e.g. 1234567)");
                textInput("shardStrId", "Shard String ID:", "", "(e.g. solomon_prod_stockpile)");
                tr(() -> {
                    tdText("Move To Host:");
                    td(() -> {
                        List<String> fqdns = balancer.getCluster().getHosts()
                            .stream()
                            .map(CoremonHost::getFqdn)
                            .sorted()
                            .collect(Collectors.toList());
                        selectFormControl("fqdn", "fqdn", fqdns, "");
                    });
                });
            });
            tag("div.form-group", () -> {
                buttonSubmitDefault("Move");
            });
        }, new Attr("action", "/balancer-old/moveShard"), new Attr("class", "form-horizontal"));
    }

    private void timeInput(String id, String name, long valueMillis, long defaultMillis) {
        textInput(id, name, formatDurationMillis(valueMillis), "(default: " + formatDurationMillis(defaultMillis) + ')');
    }

    private void intInput(String id, String name, int value, int defaultValue) {
        textInput(id, name, Integer.toString(value), String.format("(default: %d)", defaultValue));
    }

    private void doubleInput(String id, String name, double value, double defaultValue) {
        textInput(id, name, String.format("%.6f", value), String.format("(default: %.6f)", defaultValue));
    }

    private void booleanInput(String id, String name, boolean value, boolean defaultValue) {
        textInput(id, name, String.format("%s", value), String.format("(default: %s)", defaultValue));
    }

    private void textInput(String id, String name, String value, String help) {
        tr(() -> {
            thText(name);
            td(() -> {
                inputTextfieldFormControl(id, id, value);
            });
            if (StringUtils.isNotEmpty(help)) {
                td(() -> {
                    tag("span", HtmlWriter.Attr.cssClass("help-block"), () -> {
                        write(help);
                    });
                });
            } else {
                td(() -> {});
            }
        });
    }

    private void writeHostsTable(CoremonCluster cluster, ShardBalancerOptions options) {
        HostSummary[] rows = cluster.getHosts().stream()
            .map(h -> {
                boolean active = !options.getInactiveHosts().contains(h.getFqdn());
                return new HostSummary(h.getFqdn(), h.getSeenAliveTimeMillis(), active, h.getState(true));
            })
            .toArray(HostSummary[]::new);

        LoadCalc.Load clusterLoad = LoadCalc.sumLoads(
            Arrays.stream(rows)
            .map(r -> r.load)
            .toArray(LoadCalc.Load[]::new));

        for (HostSummary row : rows) {
            row.loadScore = LoadCalc.loadScore(
                row.load,
                clusterLoad,
                options.getCpuWeightFactor(),
                options.getMemoryWeightFactor(),
                options.getNetworkWeightFactor());
        }

        Column[] columns = makeHostSummaryColumns(rows);
        Comparator<HostSummary> comparator = columns[Math.abs(sortBy) - 1].comparator;
        if (sortBy < 0) {
            comparator = comparator.reversed();
        }
        Arrays.sort(rows, comparator);

        tag("table", () -> {
            tr(() -> {
                for (int index = 0; index < columns.length; index++) {
                    int i = index;
                    th(() -> {
                        columns[i].header.accept(i);
                    });
                }
            });

            int containerIndex = 0;
            int containersPerHost = 4;

            for (HostSummary h : rows) {
                boolean isOddHost = (containerIndex++ / containersPerHost) % 2 == 1;

                tr(() -> {
                    for (Column column : columns) {
                        td(() -> {
                            column.content.accept(h);
                        });
                    }
                }, isOddHost ? new Attr[] { Attr.cssClass("active") } : EMPTY_ATTRS);
            }
        }, Attr.cssClass("table simple-table2 table-hover table-condensed"));
    }

    private Column[] makeHostSummaryColumns(HostSummary[] rows) {
        boolean allActive = Arrays.stream(rows).allMatch(h -> h.active);

        return new Column[]{
            column("hostname", h -> {
                if (HostUtils.getFqdn().equals(h.fqdn)) {
                    tag("span", HtmlWriter.Attr.cssClass("glyphicon glyphicon-star"));
                } else {
                    tag("span", HtmlWriter.Attr.cssClass("glyphicon"));
                }
                write(" ");
                aHref("http://" + h.fqdn + ":" + port + "/local-shards", HostUtils.shortName(h.fqdn));
            }, Comparator.comparing(h -> h.fqdn)),

            column("Last Seen", h -> {
                if (h.seenAliveTimeMillis == 0) {
                    label("danger", "DEAD");
                } else {
                    long duration = System.currentTimeMillis() - h.seenAliveTimeMillis;
                    String durationStr = DurationUtils.toHumanReadable(duration);
                    if (duration >= options.getOfflineThresholdMillis()) {
                        label("warning", "OFFLINE");
                        write(" " + durationStr);
                    } else {
                        label("success", "ONLINE");
                        write(" " + durationStr);
                    }
                }
            }, Comparator.comparing(h -> System.currentTimeMillis() - h.seenAliveTimeMillis)),

            column("Uptime", h -> {
                write(DurationUtils.toHumanReadable(h.uptimeMillis));
            }, Comparator.comparingLong(h -> h.uptimeMillis)),

            column("Load Score", h -> {
                write(String.format("%.6f", h.loadScore));
            }, Comparator.comparingDouble(h -> h.loadScore)),

            column("CPU/ms", h -> {
                write(DataSize.shortString(TimeUnit.NANOSECONDS.toMillis(Math.round(h.load.cpuTimeNanos))));
            }, Comparator.comparingDouble(h -> h.load.cpuTimeNanos)),

            column("Metrics/count", h -> {
                write(DataSize.shortString((long) h.load.metricsCount));
            }, Comparator.comparingDouble(h -> h.load.metricsCount)),

            column("Network/bytes", h -> {
                write(DataSize.shortString((long) h.load.networkBytes));
            }, Comparator.comparingDouble(h -> h.load.networkBytes)),

            column("Shards", h -> {
                if (h.shardsSynced) {
                    label("success", "SYNCED");
                } else {
                    label("info", "SYNCING");
                }

                String value = h.shardsReady != h.shardsTotal
                    ? h.shardsReady + " / " + h.shardsTotal
                    : String.valueOf(h.shardsTotal);
                write(" " + value);
            }, Comparator.comparingDouble(h -> h.shardsReady)),

            new Column(ignore -> {
                aHref("/balancer-old/changeHostState?fqdn=ALL&active=" + !allActive, "Active");
            }, h -> {
                aHref("/balancer-old/changeHostState?fqdn=" + h.fqdn + "&active=" + !h.active, () -> {
                    if (h.active) {
                        tag("span", Attr.cssClass("active-mark glyphicon glyphicon-ok"));
                    } else {
                        tag("span", Attr.cssClass("active-mark glyphicon glyphicon-remove"));
                    }
                });
            }, Comparator.comparing(h1 -> h1.fqdn)),

            column("Kick", h -> {
                aHref("/balancer-old/kickShards?fqdn=" + h.fqdn, () -> {
                    tag("span", HtmlWriter.Attr.cssClass("glyphicon glyphicon-transfer"));
                });
            }, Comparator.comparing(h -> h.fqdn)),
        };
    }

    private Column column(String title, Consumer<HostSummary> content, Comparator<HostSummary> comparator) {
        return new Column(idx -> {
            int column = idx + 1;
            int sort;
            String glyphicon;
            if (column == Math.abs(sortBy)) {
                sort = sortBy * -1;
                if (sort > 0) {
                    glyphicon = "glyphicon glyphicon-sort-by-attributes-alt";
                } else {
                    glyphicon = "glyphicon glyphicon-sort-by-attributes";
                }
            } else {
                glyphicon = null;
                sort = column;
            }

            aHref("/balancer-old?sortBy=" + sort, () -> {
                if (glyphicon != null) {
                    tag("span", HtmlWriter.Attr.cssClass(glyphicon));
                }
                write(title);
            });
        }, content, comparator);
    }

    private static final class Column {
        private final IntConsumer header;
        private final Consumer<HostSummary> content;
        private final Comparator<HostSummary> comparator;

        Column(IntConsumer header, Consumer<HostSummary> content, Comparator<HostSummary> comparator) {
            this.header = header;
            this.content = content;
            this.comparator = comparator;
        }
    }

    /**
     * HOST SUMMARY
     */
    private static final class HostSummary {
        private final String fqdn;
        private final long uptimeMillis;
        private final long seenAliveTimeMillis;
        private final boolean active;
        private final boolean shardsSynced;
        private final LoadCalc.Load load;
        private final int shardsReady;
        private final int shardsTotal;
        private double loadScore;

        HostSummary(String fqdn, long seenAliveTimeMillis, boolean active, CoremonHost.State hostState) {
            this.fqdn = fqdn;
            this.uptimeMillis = hostState.getUptimeMillis();
            this.seenAliveTimeMillis = seenAliveTimeMillis;
            this.active = active;
            this.shardsSynced = hostState.isSynced();
            this.load = LoadCalc.sumShards(hostState.getShards().values());

            int shardsReady = 0, shardsTotal = 0;
            for (ShardLoad s : hostState.getShards().values()) {
                if (s.getState() == EShardState.INACTIVE) {
                    continue;
                }
                if (s.getState() == EShardState.READY) {
                    shardsReady++;
                }
                shardsTotal++;
            }
            this.shardsReady = shardsReady;
            this.shardsTotal = shardsTotal;
        }
    }
}
