package ru.yandex.solomon.coremon.push;

import java.io.StringWriter;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.util.CharsetUtil;

import ru.yandex.monlib.metrics.encode.spack.MetricSpackDecoder;
import ru.yandex.monlib.metrics.encode.text.MetricTextEncoder;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.model.protobuf.MetricFormat;
import ru.yandex.solomon.staffOnly.html.HtmlWriterWithCommonLibraries;
import ru.yandex.solomon.staffOnly.manager.ExtraContentParam;
import ru.yandex.solomon.staffOnly.manager.special.ExtraContent;
import ru.yandex.solomon.util.time.InstantUtils;


/**
 * @author Sergey Polovko
 */
public class LastPushedData {

    private final String host;
    private final Labels hostOptLabels;
    private final MetricFormat format;
    private final ByteBuf data;
    private final String pushedAt;
    private final boolean isPush;

    public LastPushedData(
        String host,
        Labels hostOptLabels,
        MetricFormat format,
        ByteBuf data,
        long instantMillis,
        boolean isPush)
    {
        this.host = host;
        this.hostOptLabels = hostOptLabels;
        this.format = format;
        this.data = data;
        this.pushedAt = InstantUtils.formatToSeconds(instantMillis);
        this.isPush = isPush;
    }

    public void clear() {
        data.release();
    }

    @ExtraContent("Data TEXT")
    public void extraText(ExtraContentParam p) {
        HtmlWriterWithCommonLibraries w = p.getHtmlWriter();
        w.preText(toString(format, data.slice()));
    }

    @ExtraContent("Data HEX")
    public void extraHex(ExtraContentParam p) {
        HtmlWriterWithCommonLibraries w = p.getHtmlWriter();
        w.preText(ByteBufUtil.prettyHexDump(data));
    }

    private static String toString(MetricFormat format, ByteBuf data) {
        if (format == MetricFormat.SPACK) {
            StringWriter stringWriter = new StringWriter(1024);
            try (MetricTextEncoder encoder = new MetricTextEncoder(stringWriter, true)) {
                MetricSpackDecoder decoder = new MetricSpackDecoder();
                decoder.decode(data.nioBuffer(), encoder);
            }
            return stringWriter.toString();
        } else if (format == MetricFormat.JSON) {
            return data.toString(CharsetUtil.UTF_8);
        }

        return "unsupported format " + format;
    }

    @Override
    public String toString() {
        return "LastPushedData{" +
            "host='" + host + '\'' +
            ", hostOptLabels=" + hostOptLabels +
            ", format=" + format +
            ", data=" + data.readableBytes() + " bytes" +
            ", pushedAt='" + pushedAt + '\'' +
            ", isPush=" + isPush +
            '}';
    }
}
