package ru.yandex.solomon.coremon.shards;

import java.time.Instant;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Charsets;
import com.google.common.hash.Hashing;

import ru.yandex.solomon.config.protobuf.coremon.TCoremonCreateShardConfig;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.DecimPolicy;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceProviderShardSettings;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.db.model.ShardSettings;
import ru.yandex.solomon.core.validators.IdValidator;
import ru.yandex.solomon.labels.shard.ShardKey;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.solomon.core.db.model.ShardSettings.AggregationSettings.EMPTY;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
final class ConfigMaker {

    private final ShardKey shardKey;
    private final String accountId;
    private final TCoremonCreateShardConfig conf;
    private final String clusterId;
    private final String serviceId;
    private final String shardId;

    ConfigMaker(ShardKey shardKey, String accountId, TCoremonCreateShardConfig conf) {
        this.shardKey = shardKey;
        this.accountId = accountId;
        this.conf = conf;

        this.clusterId = makeShortId(shardKey.getProject() + '_' + shardKey.getCluster(), IdValidator.MAX_CLUSTER_ID_LENGTH);
        IdValidator.ensureClusterIdValid(clusterId, "cluster");

        this.serviceId = makeShortId(shardKey.getProject() + '_' + shardKey.getService(), IdValidator.MAX_SERVICE_ID_LENGTH);
        IdValidator.ensureServiceIdValid(serviceId, "service");

        this.shardId = makeShortId(shardKey.getProject() + '_' + shardKey.getCluster() + '_' + shardKey.getService(), IdValidator.MAX_SHARD_ID_LENGTH);
        IdValidator.ensureShardIdValid(shardId, "shard");
    }

    static String makeShortId(String id, int maxLength) {
        if (id.length() <= maxLength) {
            return id;
        }

        var shortHash = Hashing.sha256().hashString(id, Charsets.UTF_8)
                .toString()
                .substring(0, 7);

        var prefix = id.substring(0, maxLength - 8);
        return prefix + "_" + shortHash;
    }

    public String getClusterId() {
        return clusterId;
    }

    public String getServiceId() {
        return serviceId;
    }

    public String getShardId() {
        return shardId;
    }

    Project makeProject(String abcService) {
        var now = Instant.now();
        return Project.newBuilder()
                .setId(shardKey.getProject())
                .setName(shardKey.getProject())
                .setAbcService(abcService)
                .setOwner(accountId)
                .setOnlyMetricNameShards(!conf.getMetricNameLabel().isEmpty())
                .setOnlyAuthRead(true)
                .setOnlyAuthPush(true)
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .setCreatedBy(accountId)
                .setUpdatedBy(accountId)
                .build();
    }

    Cluster makeCluster(String folderId) {
        var now = Instant.now();
        return Cluster.newBuilder()
                .setProjectId(shardKey.getProject())
                .setId(clusterId)
                .setName(shardKey.getCluster())
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .setCreatedBy(accountId)
                .setUpdatedBy(accountId)
                .setShardSettings(ShardSettings.of(
                        ShardSettings.Type.UNSPECIFIED,
                        null,
                        0,
                        0,
                        DecimPolicy.UNDEFINED,
                        EMPTY,
                        0
                ))
                .setFolderId(folderId)
                .build();
    }

    Service makeService(String serviceProvider, ServiceProviderShardSettings shardSettings) {
        var now = Instant.now();
        var builder = Service.newBuilder()
                .setProjectId(shardKey.getProject())
                .setId(serviceId)
                .setName(shardKey.getService())
                .setServiceProvider(serviceProvider)
                .setMetricConf(shardSettings.getMetricConf())
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .setCreatedBy(accountId)
                .setUpdatedBy(accountId)
                .setShardSettings(ShardSettings.of(
                        ShardSettings.Type.PUSH,
                        null,
                        shardSettings.getGrid(),
                        shardSettings.getMetricsTtlDays(),
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.of(true, shardSettings.getMetricConf().getAggrRules(), shardSettings.getMetricConf().isRawDataMemOnly()),
                        0
                ));

        if (conf.getIsCloud()) {
            // See SOLOMON-5449
            if (!"custom".equals(shardKey.getService()) && serviceProvider.isEmpty()) {
                builder.setShardSettings(ShardSettings.of(
                        ShardSettings.Type.PUSH,
                        null,
                        shardSettings.getGrid(),
                        30,
                        DecimPolicy.UNDEFINED,
                        ShardSettings.AggregationSettings.of(true, shardSettings.getMetricConf().getAggrRules(), shardSettings.getMetricConf().isRawDataMemOnly()),
                        0
                ));
            }
        }

        return builder.build();
    }

    Shard makeShard(Cluster cluster, Service service, int numId) {
        checkArgument(
                cluster.getProjectId().equals(service.getProjectId()),
                "project and service must be in one project, %s != %s",
                cluster.getProjectId(), service.getProjectId());

        Instant now = Instant.now();

        var builder = Shard.newBuilder()
                .setProjectId(cluster.getProjectId())
                .setId(shardId)
                .setNumId(numId)
                .setClusterId(cluster.getId())
                .setClusterName(cluster.getName())
                .setServiceId(service.getId())
                .setServiceName(service.getName())
                .setMetricNameLabel(conf.getMetricNameLabel())
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .setCreatedBy(accountId)
                .setUpdatedBy(accountId)
                .setShardSettings(ShardSettings.of(
                        ShardSettings.Type.UNSPECIFIED,
                        null,
                        0,
                        service.getShardSettings().getMetricsTtl() == 0 ? conf.getTtlDays() : 0,
                        DecimPolicy.UNDEFINED,
                        EMPTY,
                        0
                ));

        if (conf.getMaxFileMetrics() > 0) {
            builder.setMaxFileMetrics(conf.getMaxFileMetrics());
        }

        if (conf.getMaxMemOnlyMetrics() > 0) {
            builder.setMaxMemMetrics(conf.getMaxMemOnlyMetrics());
        }

        return builder.build();
    }
}
