package ru.yandex.solomon.coremon.shards;

import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.core.conf.ClusterConfDetailed;
import ru.yandex.solomon.core.conf.ServiceConfDetailed;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.db.dao.ClustersDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ServicesDao;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.model.Cluster;
import ru.yandex.solomon.core.db.model.Project;
import ru.yandex.solomon.core.db.model.Service;
import ru.yandex.solomon.core.db.model.ServiceProvider;
import ru.yandex.solomon.core.db.model.ServiceProviderShardSettings;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.labels.shard.ShardKey;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * Process flow:
 *                                                  +--------------+     +------------------+
 *                                                  |              |     |  Resolve Folder  |
 *                                              +-->+ Find Cluster +---->+        &         +--+
 *                                              |   |              |     |  Create Cluster  |  |
 *                                              |   +------+-------+     +------------------+  |
 * +--------------+     +----------------+      |          |                                   |   +--------------+
 * |              |     |                |      |          |             found                 |   |              |
 * | Find Project +---->+ Create Project +--+-->+          +-----------------------------------+-->+ Create Shard |
 * |              |     |                |  ^   |          |                                   |   |              |
 * +-------+------+     +----------------+  |   |          |                                   |   +--------------+
 *         |             found              |   |   +------+-------+     +----------------+    |
 *         +--------------------------------+   |   |              |     |                |    |
 *                                              +-->+ Find Service +---->+ Create Service +----+
 *                                                  |              |     |                |
 *                                                  +--------------+     +----------------+
 *
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
final class CreateShardProcess extends ShardProcess {

    private final ShardKey shardKey;
    private final ConfigMaker configMaker;
    @Nullable
    private final ServiceProvider serviceProvider;

    CreateShardProcess(ShardCreationContext context, SolomonConfWithContext conf, ShardKey shardKey, String accountId) {
        super(context, conf);
        this.shardKey = shardKey;

        this.configMaker = new ConfigMaker(shardKey, accountId, context.createShardConfig);
        this.serviceProvider = conf.getServiceProvider(shardKey.getService());

        doCreate()
            .whenComplete((shard, throwable) -> {
                if (throwable != null) {
                    getDoneFuture().completeExceptionally(throwable);
                } else {
                    getDoneFuture().complete(shard);
                }
            });
    }

    private CompletableFuture<Shard> doCreate() {
        return findOrCreateProject()
                .thenCompose(projectInserted -> {
                    var clusterFuture = findOrCreateCluster();
                    var serviceFuture = findOrCreateService();

                    return CompletableFuture.allOf(clusterFuture, serviceFuture)
                            .thenCompose(aVoid -> findOrCreateShard(
                                    clusterFuture.getNow(null),
                                    serviceFuture.getNow(null)));
                });
    }

    private CompletableFuture<Void> findOrCreateProject() {
        {
            Project project = getConf().getProject(shardKey.getProject());
            if (project != null) {
                return completedFuture(null);
            }
        }

        ProjectsDao projectsDao = getContext().projectsDao;
        return projectsDao.findById(shardKey.getProject())
                .thenCompose(projectO -> {
                    if (projectO.isPresent()) {
                        return completedFuture(null);
                    }

                    if (serviceProvider == null) {
                        String msg = "project creation is not allowed for unknown service provider: "
                                + shardKey.getService();
                        throw new BadRequestException(msg);
                    }

                    var project = configMaker.makeProject(serviceProvider.getAbcService());
                    // if project was not inserted then we know that project with the same id already exists
                    return projectsDao.insert(project).thenAccept(inserted -> {});
                });
    }

    private CompletableFuture<Cluster> findOrCreateCluster() {
        String clusterId = configMaker.getClusterId();

        ClusterConfDetailed clusterConf = getConf().getClusterOrNull(shardKey.getProject(), clusterId);
        if (clusterConf != null) {
            checkClusterName(clusterConf.getRaw(), shardKey);
            return completedFuture(clusterConf.getRaw());
        }

        ClustersDao clusterDao = getContext().clusterDao;
        return clusterDao.findOne(shardKey.getProject(), "", clusterId)
            .thenCompose(clusterO -> {
                if (clusterO.isPresent()) {
                    checkClusterName(clusterO.get(), shardKey);
                    return completedFuture(clusterO.get());
                }

                return resolveFolderId()
                    .thenCompose(folderId -> {
                        var cluster = configMaker.makeCluster(folderId);
                        return clusterDao.insert(cluster)
                            .thenCompose(inserted -> {
                                if (inserted) {
                                    return completedFuture(cluster);
                                }
                                return clusterDao.findOne(cluster.getProjectId(), "", cluster.getId())
                                    .thenApply(cluster2 -> {
                                        checkClusterName(cluster2.orElseThrow(), shardKey);
                                        return cluster2.orElseThrow();
                                    });
                            });
                    });
            });
    }

    private CompletableFuture<String> resolveFolderId() {
        return context.folderResolver.resolveFolderId(shardKey.getProject(), shardKey.getCluster());
    }

    private CompletableFuture<Service> findOrCreateService() {
        String serviceId = configMaker.getServiceId();

        ServiceConfDetailed serviceConf = getConf().getServiceOrNull(shardKey.getProject(), serviceId);
        if (serviceConf != null) {
            checkServiceName(serviceConf.getRaw(), shardKey);
            return completedFuture(serviceConf.getRaw());
        }

        ServicesDao serviceDao = getContext().serviceDao;
        return serviceDao.findOne(shardKey.getProject(), "", serviceId)
                .thenCompose(serviceO -> {
                    if (serviceO.isPresent()) {
                        checkServiceName(serviceO.get(), shardKey);
                        return completedFuture(serviceO.get());
                    }

                    final Service service;
                    if (serviceProvider != null) {
                        service = configMaker.makeService(serviceProvider.getId(), serviceProvider.getShardSettings());
                    } else {
                        service = configMaker.makeService("", ServiceProviderShardSettings.EMPTY);
                    }
                    return serviceDao.insert(service)
                            .thenCompose(inserted -> {
                                if (inserted) {
                                    return completedFuture(service);
                                }
                                return serviceDao.findOne(service.getProjectId(), "", service.getId())
                                        .thenApply(service2 -> {
                                            checkServiceName(service2.orElseThrow(), shardKey);
                                            return service2.orElseThrow();
                                        });
                            });
                });
    }

    private CompletableFuture<Shard> findOrCreateShard(Cluster cluster, Service service) {
        return findOrCreateShardIteration(cluster, service, 10);
    }

    private CompletableFuture<Shard> findOrCreateShardIteration(Cluster cluster, Service service, int leftAttempts) {
        String shardId = configMaker.getShardId();
        if (leftAttempts == 0) {
            return failedFuture(new IllegalStateException("cannot insert shard " + shardId));
        }

        ShardsDao shardDao = getContext().shardDao;
        return shardDao.findOne(cluster.getProjectId(), "", shardId)
                .thenCompose(shardO -> {
                    if (shardO.isPresent()) {
                        checkClusterAndServiceName(shardO.get(), shardKey);
                        return completedFuture(shardO.get());
                    }

                    int numId = getContext().shardNumIdGenerator.generateNumId(shardId);
                    var shard = configMaker.makeShard(cluster, service, numId);

                    return shardDao.insert(shard)
                            .thenCompose(inserted -> {
                                if (inserted) {
                                    return completedFuture(shard);
                                }
                                return findOrCreateShardIteration(cluster, service, leftAttempts - 1);
                            });
                });
    }

    private static void checkClusterName(Cluster cluster, ShardKey shardKey) {
        if (!cluster.getName().equals(shardKey.getCluster())) {
            throw new BadRequestException("cluster with id \"" + cluster.getId()
                    + "\" already exist with label name equal to \"" + cluster.getName()
                    + "\", but expected label name \"" + shardKey.getCluster() + "\"");
        }
    }

    private static void checkServiceName(Service service, ShardKey shardKey) {
        if (!service.getName().equals(shardKey.getService())) {
            throw new BadRequestException("service with id \"" + service.getId()
                    + "\" already exist with label name equal to \"" + service.getName()
                    + "\", but expected label name \"" + shardKey.getService() + "\"");
        }
    }

    private static void checkClusterAndServiceName(Shard shard, ShardKey shardKey) {
        String clusterName = shard.getClusterName();
        String serviceName = shard.getServiceName();
        if (!clusterName.equals(shardKey.getCluster()) || !serviceName.equals(shardKey.getService())) {
            throw new BadRequestException("shard with id \"" + shard.getId() + "\" already exist with " +
                    "\"{cluster=" + clusterName + ", service=" + serviceName + "}\", but expected " +
                    "\"{cluster=" + shardKey.getCluster() + ", service=" + shardKey.getService() + "}\"");
        }
    }
}
