package ru.yandex.solomon.coremon.shards;

import java.util.ArrayList;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.cloud.resource.resolver.FolderResolver;
import ru.yandex.solomon.cloud.resource.resolver.FolderResolverNoOp;
import ru.yandex.solomon.config.protobuf.coremon.TCoremonCreateShardConfig;
import ru.yandex.solomon.core.conf.ShardConfDetailed;
import ru.yandex.solomon.core.conf.ShardNumIdGenerator;
import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.conf.watch.SolomonConfListener;
import ru.yandex.solomon.core.db.dao.ClustersDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.ServicesDao;
import ru.yandex.solomon.core.db.dao.ShardsDao;
import ru.yandex.solomon.core.db.model.Shard;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;

import static com.google.common.base.Preconditions.checkState;
import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * Accepts shard creation requests. Keeps only one active asynchronous process per shard. All requests received
 * after the process was started will wait it's completion, instead of producing more load on underlying database.
 *
 * Also creator keeps cache of recently found or created shards to avoid making useless database lookups.
 *
 * @author Sergey Polovko
 */
@Component
@ParametersAreNonnullByDefault
public class ShardCreator implements SolomonConfListener {

    private final ShardCreatorMetrics metrics;
    private final ShardCreationContext context;
    private final ActorRunner actorRunner;
    private final ArrayListLockQueue<Event> actorEvents = new ArrayListLockQueue<>(10);
    private final Object2ObjectOpenHashMap<ShardKey, ShardProcess> inProgress = new Object2ObjectOpenHashMap<>();
    private final Cache<ShardKey, Shard> cache = CacheBuilder.newBuilder()
            .expireAfterWrite(5, TimeUnit.MINUTES)
            .maximumSize(10_000)
            .build();

    private volatile SolomonConfWithContext conf = SolomonConfWithContext.EMPTY;

    @Autowired
    public ShardCreator(
            ProjectsDao projectsDao,
            ShardsDao shardDao,
            ClustersDao clusterDao,
            ServicesDao serviceDao,
            ShardNumIdGenerator shardNumIdGenerator,
            TCoremonCreateShardConfig createShardConfig,
            Executor executor,
            MetricRegistry metricRegistry,
            Optional<FolderResolver> folderResolver)
    {
        this.metrics = new ShardCreatorMetrics(metricRegistry);
        this.context = new ShardCreationContext(
            projectsDao,
            shardDao,
            clusterDao,
            serviceDao,
            shardNumIdGenerator,
            createShardConfig,
            folderResolver.orElseGet(FolderResolverNoOp::new));
        this.actorRunner = new ActorRunner(this::act, executor);
    }

    /**
     * Asynchronously create shard if it is not exists yet.
     *
     * @param shardKey   key of shard to be created
     * @param accountId  identifier of the account, which will be recorded in
     *                   {@code createdBy} and {@code updatedBy} fields.
     * @return found or created shard.
     */
    public CompletableFuture<Shard> createShard(ShardKey shardKey, String accountId) {
        // try to find shard in last configuration
        ShardConfDetailed shard = conf.findShardOrNull(shardKey);
        if (shard != null) {
            return completedFuture(shard.getRaw());
        }

        var event = new Event.FindShard(shardKey, accountId);
        actorEvents.enqueue(event);
        actorRunner.schedule();
        return event.future;
    }

    private void act() {
        if (conf == SolomonConfWithContext.EMPTY) {
            // do not process events while we have no configuration.
            // try after some delay
            CompletableFuture.delayedExecutor(3, TimeUnit.SECONDS)
                    .execute(actorRunner::schedule);
            return;
        }

        ArrayList<Event> events = actorEvents.dequeueAll();
        for (Event event : events) {
            try {
                if (event instanceof Event.FindShard) {
                    onFindShard((Event.FindShard) event);
                } else if (event instanceof Event.CreateShard) {
                    onCreateShard((Event.CreateShard) event);
                } else if (event instanceof Event.ShardTaken) {
                    onShardTaken((Event.ShardTaken) event);
                } else if (event instanceof Event.Failure) {
                    onFailure((Event.Failure) event);
                } else {
                    throw new IllegalStateException("unknown event type " + event);
                }
            } catch (Throwable e) {
                onFailure(new Event.Failure(event.getShardKey(), e));
            }
        }
    }

    private void onFindShard(Event.FindShard event) {
        ShardKey shardKey = event.shardKey;

        {
            // (1) may be shard was recently found or created
            Shard shard = cache.getIfPresent(event.shardKey);
            if (shard != null) {
                metrics.incCacheHit();
                event.future.complete(shard);
                return;
            }
            metrics.incCacheMiss();
        }
        {
            // (2) may be we already started find or create process
            ShardProcess process = inProgress.get(shardKey);
            if (process != null) {
                process.subscribe(event.future);
                return;
            }
        }

        // (3) start find process
        var find = new FindShardProcess(context, conf, event.shardKey);
        checkState(inProgress.put(shardKey, find) == null);
        find.subscribe(event.future);
        find.getDoneFuture()
                .whenComplete((shard, throwable) -> {
                    if (throwable != null) {
                        actorEvents.enqueue(new Event.Failure(shardKey, throwable));
                    } else if (shard != null) {
                        actorEvents.enqueue(new Event.ShardTaken(shardKey, shard));
                    } else {
                        actorEvents.enqueue(new Event.CreateShard(event));
                    }
                    actorRunner.schedule();
                });
        metrics.startProcess();
    }

    private void onCreateShard(Event.CreateShard event) {
        ShardKey shardKey = event.shardKey;

        var create = new CreateShardProcess(context, conf, event.shardKey, event.accountId);
        var find = inProgress.put(shardKey, create);
        checkState(find instanceof FindShardProcess);
        create.subscribe(find.getSubscribers());
        create.getDoneFuture()
                .whenComplete((shardConf, throwable) -> {
                    if (throwable != null) {
                        actorEvents.enqueue(new Event.Failure(shardKey, throwable));
                    } else {
                        actorEvents.enqueue(new Event.ShardTaken(shardKey, shardConf));
                    }
                    actorRunner.schedule();
                });
    }

    private void onShardTaken(Event.ShardTaken event) {
        cache.put(event.shardKey, event.shard);
        metrics.setCacheSize(cache.size());

        ShardProcess process = inProgress.remove(event.shardKey);
        if (process != null) {
            process.complete(event.shard);
            metrics.stopProcessOk(System.currentTimeMillis() - process.getStartTimeMillis());
        }
    }

    private void onFailure(Event.Failure event) {
        ShardProcess process = inProgress.remove(event.shardKey);
        if (process != null) {
            process.completeExceptionally(event.throwable);
            metrics.stopProcessFail(System.currentTimeMillis() - process.getStartTimeMillis());
        }
    }

    @Override
    public void onConfigurationLoad(SolomonConfWithContext conf) {
        this.conf = conf;
    }
}
