package ru.yandex.solomon.coremon.shards;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.core.conf.SolomonConfWithContext;
import ru.yandex.solomon.core.db.model.Shard;

/**
 * Asynchronous process allowing to collect all subscribers waiting for active process to finish.
 *
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
class ShardProcess {

    private final long startTimeMillis = System.currentTimeMillis();
    final ShardCreationContext context;
    private final SolomonConfWithContext conf;
    private final List<CompletableFuture<Shard>> subscribers;
    private final CompletableFuture<Shard> doneFuture;

    public ShardProcess(ShardCreationContext context, SolomonConfWithContext conf) {
        this.context = context;
        this.conf = conf;
        this.subscribers = new ArrayList<>(1);
        this.doneFuture = new CompletableFuture<>();
    }

    public long getStartTimeMillis() {
        return startTimeMillis;
    }

    public ShardCreationContext getContext() {
        return context;
    }

    public SolomonConfWithContext getConf() {
        return conf;
    }

    public List<CompletableFuture<Shard>> getSubscribers() {
        return subscribers;
    }

    public CompletableFuture<Shard> getDoneFuture() {
        return doneFuture;
    }

    public void subscribe(List<CompletableFuture<Shard>> subscribers) {
        this.subscribers.addAll(subscribers);
    }

    public void subscribe(CompletableFuture<Shard> future) {
        subscribers.add(future);
    }

    public void complete(Shard shard) {
        for (CompletableFuture<Shard> subscriber : subscribers) {
            subscriber.complete(shard);
        }
    }

    public void completeExceptionally(Throwable throwable) {
        for (CompletableFuture<Shard> subscriber : subscribers) {
            subscriber.completeExceptionally(throwable);
        }
    }
}
