package ru.yandex.solomon.coremon.stockpile;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterators;
import com.google.common.collect.Multimap;
import io.netty.buffer.ByteBuf;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.monitoring.coremon.EShardState;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.codec.serializer.StockpileFormat;
import ru.yandex.solomon.core.db.model.ShardState;
import ru.yandex.solomon.core.urlStatus.UrlStatusTypes;
import ru.yandex.solomon.coremon.CoremonShardOpts;
import ru.yandex.solomon.coremon.CoremonShardQuota;
import ru.yandex.solomon.coremon.ProcessingResult;
import ru.yandex.solomon.coremon.aggregates.AggrHelper;
import ru.yandex.solomon.coremon.balancer.state.ShardLoad;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.MetricsCollection;
import ru.yandex.solomon.coremon.meta.db.MetabaseShardStorage;
import ru.yandex.solomon.coremon.meta.mem.MemOnlyMetricsCollection;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.stockpile.write.UnresolvedMetricData;
import ru.yandex.solomon.labels.intern.InterningLabelAllocator;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;
import ru.yandex.solomon.metrics.parser.ErrorListenerCounter;
import ru.yandex.solomon.metrics.parser.FormatListenerCounter;
import ru.yandex.solomon.metrics.parser.TreeParser;
import ru.yandex.solomon.metrics.parser.json.TreeParserJson;
import ru.yandex.solomon.metrics.parser.monitoringJson.TreeParserMonitoringJson;
import ru.yandex.solomon.metrics.parser.prometheus.RemoteWriteParser;
import ru.yandex.solomon.metrics.parser.protobuf.TreeParserProtobuf;
import ru.yandex.solomon.metrics.parser.spack.TreeParserSpack;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricFormat;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.selfmon.executors.CpuMeasureExecutor;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethodArgument;
import ru.yandex.solomon.staffOnly.manager.find.NamedObject;
import ru.yandex.solomon.staffOnly.manager.special.InstantMillis;
import ru.yandex.solomon.staffOnly.manager.special.PullHere;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueueMem;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.stockpile.client.StockpileClient;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class CoremonShardStockpile implements NamedObject {
    private static final Logger logger = LoggerFactory.getLogger(CoremonShardStockpile.class);

    private static final int INITIAL_PARSING_ACTORS_COUNT = 24;
    private static final int MAX_PARSING_ACTORS_COUNT = 48;

    private final CoremonMericConfDetailed metricConfDetailed;
    private final StockpileClient stockpileClient;
    private final MetricProcessorFactory processorFactory;
    private final TreeParser treeParser;
    public final LabelAllocator labelAllocator;
    private volatile ParsingActor[] parsingActors;
    private final AggrHelper aggrHelper;
    @PullHere
    private volatile CoremonShardOpts opts;

    private final AtomicBoolean onlyNewFormatWritesRef;
    // update at each process start
    @Nullable
    StockpileFormat serverFormat;
    private Throwable lastException;
    @InstantMillis
    private long lastExceptionInstant;

    private final Executor parsingThreadPool;
    private final Executor miscThreadPool;
    private final CoremonShardStockpileMetrics metrics;
    private final TreeParser.FormatListener formatListener;
    private final ProcessingWriter stockpileBufferedWriter;
    private final MetabaseShard metabaseShard;

    public CoremonShardStockpile(
            CoremonMericConfDetailed metricConfDetailed,
            TreeParser treeParser,
            LabelAllocator labelAllocator,
            Executor parsingThreadPool,
            Executor miscThreadPool,
            StockpileClient stockpileClient,
            MetricProcessorFactory processorFactory,
            CoremonShardOpts opts,
            ProcessingWriter stockpileBufferedWriter,
            MetabaseShard metabaseShard)
    {
        this.metricConfDetailed = metricConfDetailed;
        this.treeParser = treeParser;
        this.labelAllocator = labelAllocator;
        this.stockpileClient = stockpileClient;
        this.processorFactory = processorFactory;
        this.stockpileBufferedWriter = stockpileBufferedWriter;
        this.metabaseShard = metabaseShard;

        long urlTickMillis = opts.getFetchMillis() > 0 ? opts.getFetchMillis() : 15_000;
        this.metrics = new CoremonShardStockpileMetrics(urlTickMillis, opts.getId().getShardKey().getProject(), opts.getId().getShardId());
        this.parsingThreadPool = parsingThreadPool;
        this.miscThreadPool = new CpuMeasureExecutor(metrics.cpuTimeNanos, miscThreadPool);

        this.opts = opts;
        this.onlyNewFormatWritesRef = new AtomicBoolean(opts.isOnlyNewFormatWrites());

        this.aggrHelper = new AggrHelper(metricConfDetailed.getAggrRules(), opts.getDecimPolicyId(), labelAllocator);

        final ParsingActor[] parsingActors = new ParsingActor[INITIAL_PARSING_ACTORS_COUNT];
        for (int i = 0; i < parsingActors.length; i++) {
            parsingActors[i] = new ParsingActor(i, metabaseShard);
        }
        this.parsingActors = parsingActors;

        this.formatListener = new FormatListenerCounter(metrics.oldFormatMetrics, metrics.newFormatMetrics);

        setOpts(opts);
    }

    @ManagerMethod
    private void changeParsingActorsCount(@ManagerMethodArgument(name = "count") int count) {
        if (count <= 0 || count > MAX_PARSING_ACTORS_COUNT) {
            throw new IllegalArgumentException("invalid count value: " + count);
        }

        final ParsingActor[] parsingActors = this.parsingActors;
        if (count == parsingActors.length) {
            return;
        }

        final ParsingActor[] newParsingActors = Arrays.copyOf(parsingActors, count);
        if (count > parsingActors.length) {
            for (int i = parsingActors.length; i < count; i++) {
                newParsingActors[i] = new ParsingActor(i, metabaseShard);
            }
        }
        this.parsingActors = newParsingActors;
    }

    @Nonnull
    public CoremonShardStockpileMetrics updateAndGetMetrics(MetabaseShard metabaseShard) {
        MetabaseShardStorage storage = metabaseShard.getStorage();
        MemOnlyMetricsCollection memOnlyMetrics = storage.getMemOnlyMetrics();
        MetricsCollection<CoremonMetric> fileMetrics = storage.getFileMetrics();
        CoremonShardQuota quota = opts.getQuota();

        // Processing
        long queueSize = 0;
        var parsingActors = this.parsingActors;
        for (var parsingActor : parsingActors) {
            queueSize += parsingActor.queue.estimateSize();
        }
        metrics.queueSize.set(queueSize);

        // Metabase
        metrics.indexSize.set(fileMetrics.searchIndexSize());
        metrics.indexCacheSize.set(fileMetrics.searchIndexCacheSize());
        metrics.fileMetrics.set(fileMetrics.size());
        metrics.fileMetricsLimit.set(quota.getMaxFileMetrics());

        metrics.inMemMetrics.set(memOnlyMetrics.size());
        metrics.inMemMetricsLimit.set(quota.getMaxMemMetrics());
        assert metrics.perUrlMetricsLimit != null : "perUrlMetricsLimit should be != null for non-total metrics";
        metrics.perUrlMetricsLimit.set(quota.getMaxMetricsPerUrl());
        metrics.storageWriteQueueMetrics.set(storage.getWriteQueueMetrics());
        return metrics;
    }

    public CoremonShardStockpileMetrics getMetrics() {
        return metrics;
    }

    public Set<String> getAggrUseLabelNames() {
        return metricConfDetailed.getAggrUseLabelNames();
    }

    public boolean isHostRequired() {
        return metricConfDetailed.isHostRequired();
    }

    @Nonnull
    public ShardKey getShardKey() {
        return opts.getId().getShardKey();
    }

    @Nonnull
    public String getId() {
        return opts.getId().getShardId();
    }

    public String getFolderId() {
        return opts.getFolderId();
    }

    public int getNumId() {
        return opts.getId().getNumId();
    }

    public Throwable getLastException() {
        return lastException;
    }

    public long getLastExceptionInstant() {
        return lastExceptionInstant;
    }

    public CompletableFuture<ProcessingResult> pushPage(TwoResponsesRequest request) {
        if (!isReady()) {
            // TODO: count
            // TODO: wait
            return completedFuture(ProcessingResult.fail(UrlStatusType.SHARD_NOT_INITIALIZED, "shard is " + getState()));
        }

        if (opts.getShardState() == ShardState.INACTIVE || opts.getShardState() == ShardState.READ_ONLY) {
            return completedFuture(ProcessingResult
                .fail(UrlStatusType.SHARD_IS_NOT_WRITABLE, "shard is in " + opts.getShardState() + " state"));
        }

        var parsingActor = findLeastLoadedParsingActor();
        if (parsingActor == null) {
            return completedFuture(ProcessingResult.fail(UrlStatusType.UNKNOWN_ERROR, "cannot find actor to process push"));
        }

        parsingActor.push(request);
        return request;
    }

    @Nullable
    private ParsingActor findLeastLoadedParsingActor() {
        final ParsingActor[] parsingActors = this.parsingActors;

        int minQueueSize = Integer.MAX_VALUE;
        ParsingActor result = null;
        for (ParsingActor p : parsingActors) {
            int queueSize = p.queue.estimateSize();
            if (minQueueSize > queueSize) {
                minQueueSize = queueSize;
                result = p;
            }
        }
        return result;
    }

    public void addMemoryInfo(MemoryBySubsystem r) {
        String coremonShard = CoremonShardStockpile.class.getSimpleName();

        long queueSize = 0;
        long inProcessingSize = 0;
        var parsingActors = this.parsingActors;
        for (var parsingActor : parsingActors) {
            queueSize += parsingActor.memorySizeInQueue();
            inProcessingSize += parsingActor.memorySizeInProcessing();
        }

        r.addMemory(coremonShard + ".queue", queueSize);
        r.addMemory(coremonShard + ".processing", inProcessingSize);

        if (labelAllocator instanceof InterningLabelAllocator) {
            long labelsSize = ((InterningLabelAllocator) labelAllocator).memorySizeIncludingSelf();
            r.addMemory(coremonShard + ".labelAllocator", labelsSize);
        }
    }

    CompletableFuture<Void> postProcessUnresolvedMetrics(CoremonShardStockpileResolveHelper resolveHelper) {
        List<CoremonMetric> updatedMetrics = resolveHelper.getUpdate();
        List<UnresolvedMetricData> newMetrics = resolveHelper.getUnresolvedMetricData();
        Multimap<String, UnresolvedMetricData> newAggregates = resolveHelper.getUnresolvedAggrMetricsDataByHost();

        // (1) write data into Metabase
        CompletableFuture<Void> writeFuture = metabaseShard.getStorage().write(updatedMetrics, newMetrics, newAggregates);
        if (!writeFuture.isDone()) {
            metrics.metabaseWrites.forFuture(writeFuture);
        }

        return writeFuture.thenComposeAsync(unit -> {
            // (2) write data into Stockpile
            var writeHelper = stockpileBufferedWriter.newHelper(getNumId(), metrics);
            completePartiallyParsedDocument(resolveHelper, writeHelper);
            return metrics.stockpileWrites.wrapFuture(writeHelper.write());
        }, miscThreadPool);
    }

    private void completePartiallyParsedDocument(
        CoremonShardStockpileResolveHelper resolveHelper,
        CoremonShardStockpileWriteHelper writeHelper)
    {
        var tempPoint = new AggrPoint();

        var it = Iterators.concat(
                resolveHelper.getUnresolvedMetricData().iterator(),
                resolveHelper.getUnresolvedAggrMetricsDataByHost().values().iterator());
        while (it.hasNext()) {
            var data = it.next();
            try (CoremonMetric metric = resolveHelper.tryResolveMetric(data.getLabels(), data.getType())) {
                if (metric == null) {
                    // metric can be null due concurrent changes in CoremonShardStockpileMetricCollection
                    // produced by manager methods (see annotated with @ManagerMethod methods)
                    continue;
                }

                for (int i = 0, count = data.pointsCount(); i < count; i++) {
                    data.get(i, tempPoint);
                    writeHelper.addPoint(
                            metric.getShardId(),
                            metric.getLocalId(),
                            tempPoint,
                            opts.getDecimPolicyId().getNumber(),
                            data.getWriteType());
                }
            }
        }
    }

    private void completeRequests(List<TwoResponsesRequest> requests, @Nullable Throwable throwable) {
        if (throwable != null) {
            rememberLastException(throwable);

            UrlStatusType statusType = UrlStatusTypes.classifyError(throwable);
            if (statusType == UrlStatusType.UNKNOWN_ERROR || statusType == UrlStatusType.UNKNOWN_STATUS_TYPE) {
                logger.error("cannot write data in {}", opts.getId().getShardId(), throwable);
            }

            for (TwoResponsesRequest request : requests) {
                metrics.incUrlProcessResult(statusType);
                request.completeExceptionally(throwable);
            }
        } else {
            for (TwoResponsesRequest request : requests) {
                Preconditions.checkNotNull(request.delayedResult);
                request.complete(request.delayedResult);
                metrics.incUrlProcessResult(request.delayedResult.getStatusType());

                ErrorListenerCounter parseErrors = request.delayedResult.getParseErrors();
                if (parseErrors != null) {
                    for (TreeParser.InvalidMetricReason type : TreeParser.InvalidMetricReason.values()) {
                        long errors = parseErrors.invalidMetrics.get(type);
                        if (errors != 0) {
                            metrics.addMetricParseError(type, errors);
                        }
                    }
                }
            }
        }
    }

    ProcessingResult processTwoPages(
        Labels commonLabels, Labels commonOptLabels,
        MetricFormat metricFormat,
        ByteBuf response, long instantMillis,
        ByteBuf prevResponse, long prevInstantMillis,
        boolean isPush, boolean onlyNewFormatWrites,
        CoremonShardStockpileResolveHelper resolveHelper,
        CoremonShardStockpileWriteHelper writeHelper)
    {
        metrics.receive(metricFormat, response.readableBytes());
        if (prevInstantMillis != 0 && instantMillis - prevInstantMillis != opts.getFetchMillis()) {
            metrics.documentsNotMatchInterval.inc();
            if (logger.isDebugEnabled()) {
                logger.debug("not matched interval, shard: {}, fetch interval: {}, period: {}, prevTs: {}, ts: {}",
                        getId(),
                        DurationUtils.formatDurationMillis(opts.getFetchMillis()),
                        DurationUtils.formatDurationMillis(instantMillis - prevInstantMillis),
                        Instant.ofEpochMilli(prevInstantMillis),
                        Instant.ofEpochMilli(instantMillis));
            }
        }

        boolean onlyNewFormatWritesInShard = onlyNewFormatWrites
            || onlyNewFormatWritesRef.get()
            || metricFormat == MetricFormat.MONITORING_JSON;

        TreeParser treeParser = treeParser(metricFormat);
        Supplier<MetricsPrevValues> prevValues = MetricsPrevValuesProcessor.lazyParsed(
            commonLabels,
            prevResponse,
            prevInstantMillis,
            treeParser,
            onlyNewFormatWritesInShard);

        ErrorListenerCounter errors = new ErrorListenerCounter();

        try (MetricProcessor metricProcessor = processorFactory.create(
            opts.getId(),
            aggrHelper, resolveHelper, writeHelper,
            metabaseShard.getMemOnlyMetrics(), commonOptLabels,
            isPush,
            instantMillis,
            prevInstantMillis,
            prevValues,
            opts.getGridMillis(),
            opts.getFetchMillis(),
            opts.getDecimPolicyId(),
            opts.getQuota(),
            metricConfDetailed.isRawDataMemOnly(),
            errors,
            opts.getValidationMode()))
        {
            try {
                treeParser.parseAndProcess(commonLabels, response, metricProcessor, errors, formatListener, onlyNewFormatWritesInShard);
                var status = metricProcessor.getStatus();
                var metricStats = metricProcessor.getMetricStats();
                return ProcessingResult.maybeOk(
                        status.getUrlStatusType(),
                        status.getMessage(),
                        metricStats.getCount(),
                        errors);
            } finally {
                var metricStats = metricProcessor.getMetricStats();
                metrics.metricsKnown.add(metricStats.getCount() - metricStats.getUnknown());
                metrics.metricsUnknown.add(metricStats.getUnknown());
                metrics.metricsTotal.add(metricStats.getCount());
                metrics.parsedMetrics.mark(metricStats.getCount());
                metrics.timeSeriesCount.add(metricStats.getTimeseriesCount());
                metrics.timeSeriesMoreThenOnePointCount.add(metricStats.getTimeseriesMoreOnePointCount());
                metrics.addMetricTypes(metricStats.getTypes());
                assert metrics.perUrlMetricsMax != null : "perUrlMetricsMax should be != null for non-total metrics";
                metrics.perUrlMetricsMax.record(metricStats.getCount());
            }
        }
    }

    private TreeParser treeParser(MetricFormat metricFormat) {
        return switch (metricFormat) {
            case JSON -> new TreeParserJson(labelAllocator, opts.getMetricNameLabel());
            case SPACK -> new TreeParserSpack(labelAllocator, opts.getMetricNameLabel());
            case MONITORING_JSON -> new TreeParserMonitoringJson(labelAllocator, opts.getMetricNameLabel());
            case PROMETHEUS -> new RemoteWriteParser(labelAllocator, opts.getMetricNameLabel());
            case PROTOBUF -> new TreeParserProtobuf(labelAllocator, opts.getMetricNameLabel());
            default -> this.treeParser;
        };
    }

    public void stop() {
    }

    @Override
    public String namedObjectId() {
        return Integer.toUnsignedString(opts.getId().getNumId());
    }

    @Override
    public String namedObjectClassId() {
        return CoremonShardStockpile.class.getName();
    }

    private void rememberLastException(Throwable x) {
        lastException = x;
        lastExceptionInstant = System.currentTimeMillis();
        metrics.errors.inc();
    }

    public ShardLoad getLoad() {
        final long uptimeMillis = metabaseShard.getUptimeMillis();
        final long cpuLoad = Math.round(metrics.cpuTimeNanos.getRate(TimeUnit.SECONDS));
        final long memoryLoad = metabaseShard.fileAndMemOnlyMetrics();
        final long networkLoad = Math.round(metrics.bytesReceived.getRate(TimeUnit.SECONDS));
        final long parsedMetrics = Math.round(metrics.parsedMetrics.getRate(TimeUnit.SECONDS));
        return new ShardLoad(getNumId(), getState(), uptimeMillis, cpuLoad, memoryLoad, networkLoad, parsedMetrics);
    }

    public boolean isReady() {
        return metabaseShard.isLoaded();
    }

    public EShardState getState() {
        return metabaseShard.getState();
    }

    private class ParsingActor {
        private final int actorNo;
        private final MetabaseShard metabaseShard;

        private final String wtd;
        // queues
        private final ArrayListLockQueueMem<TwoResponsesRequest> queue = new ArrayListLockQueueMem<>();
        private final ParsingActorRunner actorRunner = new ParsingActorRunner(
            5,
            this::act,
            parsingThreadPool,
            metrics.cpuTimeNanos);
        private final AtomicLong inProcessMemSize = new AtomicLong();


        ParsingActor(int actorNo, MetabaseShard metabaseShard) {
            this.actorNo = actorNo;
            this.wtd = getId() + " processing in actor #" + actorNo;
            this.metabaseShard = metabaseShard;
        }

        private CompletableFuture<?> act() {
            WhatThreadDoes.Handle h = WhatThreadDoes.push(wtd);
            try {
                return actInner();
            } catch (Throwable t) {
                return CompletableFuture.failedFuture(t);
            } finally {
                h.popSafely();
            }
        }

        private CompletableFuture<Void> actInner() {
            var formats = stockpileClient.getCompatibleCompressFormat();
            if (formats.isEmpty() || !isReady()) {
                // TODO: count

                RuntimeException x = new RuntimeException(
                    "shard " + getId() + " not fully ready: "
                        + (formats.isEmpty() ? "stockpile unavailable" : "not loaded"),
                    metabaseShard.getStorage().getLastLoadException());

                ArrayList<TwoResponsesRequest> batches = queue.dequeueAll();
                for (TwoResponsesRequest batch : batches) {
                    batch.completeExceptionally(x);
                }

                return completedFuture(null);
            }

            CoremonShardStockpile.this.serverFormat = StockpileFormat.byNumberOrCurrent(formats.upperEndpoint());

            // TODO: per actor limits should probably be adjusted
            //int processMax = Math.max(metricCollection.size() / 4, 100_000);
            ArrayList<TwoResponsesRequest> clientRequests0 = queue.dequeueAll();

            if (clientRequests0.isEmpty()) {
                return completedFuture(null);
            }

            inProcessMemSize.addAndGet(clientRequests0.stream()
                .mapToLong(TwoResponsesRequest::memorySizeIncludingSelf)
                .sum());

            CoremonShardStockpileResolveHelper resolveHelper = new CoremonShardStockpileResolveHelper(
                    metabaseShard.getStorage().getFileMetrics(),
                    opts.getQuota());

            List<TwoResponsesRequest> requestsWithAllKnownMetrics = new ArrayList<>(clientRequests0.size() / 2);
            List<TwoResponsesRequest> requestsWithUnknownMetrics = new ArrayList<>(clientRequests0.size() / 2);

            try {
                var writeHelper = stockpileBufferedWriter.newHelper(getNumId(), metrics);
                for (TwoResponsesRequest twoResponsesRequest : clientRequests0) {
                    try {
                        int unresolvedMetricCount = resolveHelper.getUnresolvedMetricCount();
                        ProcessingResult r = processTwoPages(
                            twoResponsesRequest.commonLabels, twoResponsesRequest.commonOptLabels,
                            twoResponsesRequest.metricFormat,
                            twoResponsesRequest.request, twoResponsesRequest.requestInstant,
                            twoResponsesRequest.prevRequest, twoResponsesRequest.prevInstant,
                            twoResponsesRequest.isPush,
                            twoResponsesRequest.onlyNewFormatWrites,
                            resolveHelper,
                            writeHelper);

                        if (resolveHelper.getUnresolvedMetricCount() > unresolvedMetricCount) {
                            requestsWithUnknownMetrics.add(twoResponsesRequest);
                        } else {
                            requestsWithAllKnownMetrics.add(twoResponsesRequest);
                        }
                        twoResponsesRequest.delayedResult = r;
                    } catch (Throwable e) {
                        rememberLastException(e);
                        metrics.incUrlProcessResult(UrlStatusTypes.classifyError(e));
                        twoResponsesRequest.completeExceptionally(e);
                    } finally {
                        inProcessMemSize.addAndGet(-twoResponsesRequest.memorySizeIncludingSelf());
                        twoResponsesRequest.release();
                    }
                }

                metrics.processQueueStarts.inc();

                // complete requests with all known metrics right after
                // Stockpile processed them
                CompletableFuture<Void> knownWritesFuture = writeHelper.write()
                    .whenComplete((unit, throwable) -> {
                        completeRequests(requestsWithAllKnownMetrics, throwable);
                    });
                metrics.stockpileWrites.forFuture(knownWritesFuture);

                // requests with some unresolved metrics will wait more, because
                // we have to save them into Metabase before
                long resolveHelperMemorySize = resolveHelper.memorySizeIncludingSelf();
                inProcessMemSize.addAndGet(resolveHelperMemorySize);
                postProcessUnresolvedMetrics(resolveHelper)
                    .whenComplete((x, e) -> {
                        completeRequests(requestsWithUnknownMetrics, e);
                        inProcessMemSize.addAndGet(-1 * resolveHelperMemorySize);
                    });

                return knownWritesFuture.handle((x, e) -> null);
            } catch (Throwable t) {
                completeRequests(clientRequests0, t);
                return CompletableFuture.failedFuture(new RuntimeException("unexpected exception while write", t));
            }
        }

        public void push(TwoResponsesRequest request) {
            queue.enqueue(request);
            actorRunner.schedule();
        }

        @Override
        public String toString() {
            return "ParsingActor{" +
                "actorNo=" + actorNo +
                ", queue=" + queue +
                ",..." +
                '}';
        }

        public long memorySizeInQueue() {
            return queue.memorySizeIncludingSelf();
        }

        public long memorySizeInProcessing() {
            return inProcessMemSize.get();
        }
    }

    public CoremonShardOpts getOpts() {
        return opts;
    }

    public void setOpts(CoremonShardOpts opts) {
        this.opts = opts;
        this.onlyNewFormatWritesRef.set(opts.isOnlyNewFormatWrites());
    }
}
