package ru.yandex.solomon.coremon.stockpile;

import java.time.Duration;
import java.util.EnumMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.Metric;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.meter.ExpMovingAverage;
import ru.yandex.monlib.metrics.meter.MaxMeter;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.metrics.parser.TreeParser.InvalidMetricReason;
import ru.yandex.solomon.model.protobuf.MetricFormat;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.selfmon.counters.EnumMetrics;
import ru.yandex.solomon.util.collection.enums.EnumMapToInt;


/**
 * @author Sergey Polovko
 */
public class CoremonShardStockpileMetrics {

    public static final int COUNT = 50; // estimated

    private static final MetricType[] METRIC_TYPES = MetricType.values();

    public enum WriteToStorageResult {
        SUCCESS,
        ERROR,
    }

    private final MetricRegistry registry;
    private final String prefix;

    final Meter cpuTimeNanos;
    final Meter bytesReceived;
    final Rate documentsNotMatchInterval;
    final Rate bytesWritten;
    final Meter parsedMetrics;
    final ConcurrentMap<MetricFormat, Rate> docByFormat = new ConcurrentHashMap<>();

    final GaugeInt64 indexSize;
    final GaugeInt64 indexCacheSize;
    final GaugeInt64 queueSize;
    final GaugeInt64 storageWriteQueueMetrics;
    final GaugeInt64 fileMetrics;
    final GaugeInt64 fileMetricsLimit;
    final GaugeInt64 inMemMetrics;
    final GaugeInt64 inMemMetricsLimit;

    @Nullable
    final MaxMeter perUrlMetricsMax;
    @Nullable
    final GaugeInt64 perUrlMetricsLimit;

    final Rate processQueueStarts;
    final AsyncMetrics stockpileWrites;
    final AsyncMetrics metabaseWrites;

    private final EnumMap<WriteToStorageResult, Rate> writeResults;
    private final EnumMap<MetricType, Rate> metricTypes;
    private final ConcurrentMap<InvalidMetricReason, Rate> metricsParseErrors;
    private final ConcurrentMap<UrlStatusType, Rate> urlProcessResults;
    private final Rate urlProcessResultsOk;
    public final Rate errors = new Rate();

    final Rate pushMetricsToStorage;
    final Rate metricsKnown;
    final Rate metricsUnknown;
    final Rate metricsTotal;

    final Rate oldFormatMetrics;
    final Rate newFormatMetrics;
    final Rate timeSeriesCount;
    final Rate timeSeriesMoreThenOnePointCount;

    public CoremonShardStockpileMetrics(String projectId, String shardId) {
        this(0, projectId, shardId);
    }

    public CoremonShardStockpileMetrics(long urlTickMillis, String projectId, String shardId) {
        this.registry = new MetricRegistry(Labels.of("projectId", projectId, "shardId", shardId));
        this.prefix = "engine.";

        cpuTimeNanos = registry.fiveMinutesMeter(prefix + "cpuTimeNanos");
        bytesReceived = registry.fiveMinutesMeter(prefix + "bytesReceived");
        bytesWritten = registry.rate(prefix + "bytesWritten");
        parsedMetrics = Meter.of(ExpMovingAverage.fiveMinutes());
        documentsNotMatchInterval = registry.rate(prefix + "not_match_interval");

        indexSize = registry.gaugeInt64(prefix + "indexSize");
        indexCacheSize = registry.gaugeInt64(prefix + "indexCacheSize");
        queueSize = registry.gaugeInt64(prefix + "queueSize");
        storageWriteQueueMetrics = registry.gaugeInt64(prefix + "storageWriteQueueSensors");
        fileMetrics = registry.gaugeInt64(prefix + "fileSensors");
        fileMetricsLimit = registry.gaugeInt64(prefix + "fileSensorsLimit");
        inMemMetrics = registry.gaugeInt64(prefix + "inMemSensors");
        inMemMetricsLimit = registry.gaugeInt64(prefix + "inMemSensorsLimit");
        if (urlTickMillis > 0) {
            // exclude for total metrics
            // combine is also not done for this metrics
            perUrlMetricsMax = registry.maxMeter(Duration.ofMillis(urlTickMillis), prefix + "perUrlSensorsMax");
            perUrlMetricsLimit = registry.gaugeInt64(prefix + "perUrlSensorsLimit");
        } else {
            perUrlMetricsMax = null;
            perUrlMetricsLimit = null;
        }

        processQueueStarts = registry.rate(prefix + "processQueueStarts");
        stockpileWrites = new AsyncMetrics(registry, prefix + "stockpileWrites");
        metabaseWrites = new AsyncMetrics(registry, prefix + "metabaseWrites");

        writeResults = EnumMetrics.rates(WriteToStorageResult.class, registry, prefix + "sensorProcessResult", "type");
        metricsParseErrors = new ConcurrentHashMap<>();

        urlProcessResultsOk = newUrlProcessResultRate(UrlStatusType.OK);
        urlProcessResults = new ConcurrentHashMap<>();
        urlProcessResults.put(UrlStatusType.OK, urlProcessResultsOk);

        pushMetricsToStorage = registry.rate(prefix + "pushSensorsToStorage");
        metricsUnknown = registry.rate(prefix + "sensorsUnknown");
        metricsKnown = registry.rate(prefix + "sensorsKnown");
        metricsTotal = registry.rate(prefix + "sensorsTotal");
        timeSeriesCount = registry.rate(prefix + "timeseriesCount");
        timeSeriesMoreThenOnePointCount = registry.rate(prefix + "timeseriesMoreThenOnePointCount");
        newFormatMetrics = registry.rate(prefix + "newFormatSensors");
        oldFormatMetrics = registry.rate(prefix + "oldFormatSensors");
        metricTypes = EnumMetrics.rates(MetricType.class, registry, prefix + "metricTypes", "type");
    }

    public void receive(MetricFormat format, int bytes) {
        bytesReceived.mark(bytes);
        var byFormat = docByFormat.get(format);
        if (byFormat == null) {
            byFormat = docByFormat.computeIfAbsent(format, ignore -> new Rate());
        }
        byFormat.inc();
    }

    public void addMetricParseError(InvalidMetricReason reason, long delta) {
        metricsParseErrors.computeIfAbsent(reason, this::newMetricParseError).add(delta);
    }

    public void incUrlProcessResult(UrlStatusType type) {
        if (type == UrlStatusType.OK) {
            urlProcessResultsOk.inc();
        } else {
            urlProcessResults.computeIfAbsent(type, this::newUrlProcessResultRate).inc();
        }
    }

    private Rate newUrlProcessResultRate(UrlStatusType t) {
        Labels labels = Labels.of("type", t.name());
        return registry.rate(prefix + "urlProcessResult", labels);
    }

    private Rate newMetricParseError(InvalidMetricReason r) {
        Labels labels = Labels.of("type", r.name());
        return registry.rate(prefix + "sensorParseError", labels);
    }

    public MetricRegistry getRegistry() {
        return registry;
    }

    public void addPushMetricsToStorage(long pointCount) {
        pushMetricsToStorage.add(pointCount);
    }

    public void addMetricProcessResults(WriteToStorageResult result, long count, long bytesWritten) {
        writeResults.get(result).add(count);
        this.bytesWritten.add(bytesWritten);
    }

    public long getMetricsTotal() {
        return metricsTotal.get();
    }

    public void addMetricTypes(EnumMapToInt<MetricType> types) {
        for (MetricType metricType : METRIC_TYPES) {
            int count = types.get(metricType);
            if (count != 0) {
                metricTypes.get(metricType).add(count);
            }
        }
    }

    public void combine(CoremonShardStockpileMetrics metrics) {
        cpuTimeNanos.combine(metrics.cpuTimeNanos);
        bytesReceived.combine(metrics.bytesReceived);
        bytesWritten.combine(metrics.bytesWritten);
        documentsNotMatchInterval.combine(metrics.documentsNotMatchInterval);

        indexSize.combine(metrics.indexSize);
        indexCacheSize.combine(metrics.indexCacheSize);
        queueSize.combine(metrics.queueSize);
        storageWriteQueueMetrics.combine(metrics.storageWriteQueueMetrics);
        fileMetrics.combine(metrics.fileMetrics);
        fileMetricsLimit.combine(metrics.fileMetricsLimit);
        inMemMetrics.combine(metrics.inMemMetrics);
        inMemMetricsLimit.combine(metrics.inMemMetricsLimit);

        processQueueStarts.combine(metrics.processQueueStarts);
        stockpileWrites.combine(metrics.stockpileWrites);
        metabaseWrites.combine(metrics.metabaseWrites);

        EnumMetrics.combineRates(WriteToStorageResult.class, writeResults, metrics.writeResults);

        for (Map.Entry<UrlStatusType, Rate> e : metrics.urlProcessResults.entrySet()) {
            urlProcessResults.computeIfAbsent(e.getKey(), this::newUrlProcessResultRate)
                .combine(e.getValue());
        }

        for (Map.Entry<InvalidMetricReason, Rate> e : metrics.metricsParseErrors.entrySet()) {
            metricsParseErrors.computeIfAbsent(e.getKey(), this::newMetricParseError)
                .combine(e.getValue());
        }

        pushMetricsToStorage.combine(metrics.pushMetricsToStorage);
        metricsKnown.combine(metrics.metricsKnown);
        metricsUnknown.combine(metrics.metricsUnknown);
        metricsTotal.combine(metrics.metricsTotal);
        timeSeriesCount.combine(metrics.timeSeriesCount);
        timeSeriesMoreThenOnePointCount.combine(metrics.timeSeriesMoreThenOnePointCount);

        oldFormatMetrics.combine(metrics.oldFormatMetrics);
        newFormatMetrics.combine(metrics.newFormatMetrics);

        EnumMetrics.combineRates(MetricType.class, metricTypes, metrics.metricTypes);
    }

    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer, boolean verbose) {
        if (verbose) {
            registry.append(tsMillis, commonLabels, consumer);
        } else {
            // In Cloud installation we have a lot of Coremon shards and exporting
            // all metrics about each shard is too much. So here we export only really
            // necessary metrics (e.g. quota metrics)
            if (fileMetrics.get() > 0) {
                appendMetric(tsMillis, commonLabels, consumer, "fileSensors", fileMetrics);
                appendMetric(tsMillis, commonLabels, consumer, "fileSensorsLimit", fileMetricsLimit);
            }

            appendMetric(tsMillis, commonLabels, consumer, "perUrlSensorsMax", perUrlMetricsMax);
            appendMetric(tsMillis, commonLabels, consumer, "perUrlSensorsLimit", perUrlMetricsLimit);

            if (bytesReceived.getRate(TimeUnit.SECONDS) > 0.0) {
                appendMetric(tsMillis, commonLabels, consumer, "bytesReceived", bytesReceived);
                appendMetric(tsMillis, commonLabels, consumer, "cpuTimeNanos", cpuTimeNanos);
                appendMetric(tsMillis, commonLabels, consumer, "sensorsTotal", metricsTotal);
                appendMetric(tsMillis, commonLabels, consumer, "pushSensorsToStorage", pushMetricsToStorage);
            }
        }
    }

    private void appendMetric(long tsMillis, Labels commonLabels, MetricConsumer c, String name, @Nullable Metric metric) {
        if (metric == null) {
            return;
        }
        c.onMetricBegin(metric.type());
        {
            c.onLabelsBegin(commonLabels.size() + registry.getCommonLabels().size() + 1);
            {
                commonLabels.forEach(c::onLabel);
                registry.getCommonLabels().forEach(c::onLabel);
                c.onLabel("sensor", prefix + name);
            }
            c.onLabelsEnd();
            metric.accept(tsMillis, c);
        }
        c.onMetricEnd();
    }
}
