package ru.yandex.solomon.coremon.stockpile;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.CoremonShardQuota;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.MetricsCollection;
import ru.yandex.solomon.coremon.stockpile.write.UnresolvedMetricData;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;


/**
 * @author Maksim Leonov (nohttp@)
 *
 * This class is used to collect all the changes made to the database and then to commit it
 */
@NotThreadSafe
public class CoremonShardStockpileResolveHelper implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(CoremonShardStockpileResolveHelper.class);

    private final MetricsCollection<CoremonMetric> metricCollection;
    private final CoremonShardQuota quota;
    private final List<CoremonMetric> update = new ArrayList<>();
    private final List<UnresolvedMetricData> unresolvedMetricData = new ArrayList<>();
    private final Multimap<String, UnresolvedMetricData> unresolvedAggrMetricsDataByHost = ArrayListMultimap.create();
    private int unresolvedMetricCount = 0;

    public CoremonShardStockpileResolveHelper(MetricsCollection<CoremonMetric> metricCollection, CoremonShardQuota quota) {
        this.metricCollection = metricCollection;
        this.quota = quota;
    }

    @Nullable
    public CoremonMetric tryResolveMetric(Labels labels, @Nullable MetricType type) {
        CoremonMetric existingMetric = metricCollection.getOrNull(labels);
        if (existingMetric == null) {
            return null;
        }
        updateExistingMetric(type, existingMetric);
        return existingMetric;
    }

    private void updateExistingMetric(@Nullable MetricType type, CoremonMetric existingMetric) {
        if (existingMetric.getType() != type && type != null) {
            existingMetric.setType(type);
            update.add(new FileCoremonMetric(existingMetric));
        }
    }

    public void addMetricData(UnresolvedMetricData metricData) {
        unresolvedMetricData.add(metricData);
        unresolvedMetricCount++;
    }

    public void addAggrData(String targetHost, UnresolvedMetricData metricPoint) {
        unresolvedAggrMetricsDataByHost.put(targetHost, metricPoint);
        unresolvedMetricCount++;
    }

    public List<UnresolvedMetricData> getUnresolvedMetricData() {
        return unresolvedMetricData;
    }

    public Multimap<String, UnresolvedMetricData> getUnresolvedAggrMetricsDataByHost() {
        return unresolvedAggrMetricsDataByHost;
    }

    public List<CoremonMetric> getUpdate() {
        return update;
    }

    public int getUnresolvedMetricCount() {
        return unresolvedMetricCount;
    }

    public int size() {
        return metricCollection.size();
    }

    public boolean reachQuota() {
        return metricCollection.size() > quota.getMaxFileMetrics();
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;

        size += MemoryCounter.arraySize(update.size(), MemoryCounter.OBJECT_POINTER_SIZE);
        size += update.stream()
            .mapToLong(MemMeasurable::memorySizeIncludingSelf)
            .sum();

        size += MemoryCounter.hashMapSize(unresolvedAggrMetricsDataByHost.size());
        return size;
    }
}
