package ru.yandex.solomon.coremon.stockpile;

import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.series.TimeSeries;
import ru.yandex.solomon.core.db.model.ValidationMode;
import ru.yandex.solomon.core.urlStatus.UrlStatusTypeException;
import ru.yandex.solomon.coremon.CoremonShardQuota;
import ru.yandex.solomon.coremon.aggregates.AggrHelper;
import ru.yandex.solomon.coremon.aggregates.AggrMetrics;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.mem.MemOnlyCoremonMetric;
import ru.yandex.solomon.coremon.meta.mem.MemOnlyMetricsCollection;
import ru.yandex.solomon.coremon.stockpile.write.Points;
import ru.yandex.solomon.coremon.stockpile.write.UnresolvedMetricPoint;
import ru.yandex.solomon.coremon.stockpile.write.UnresolvedMetricTimeSeries;
import ru.yandex.solomon.metrics.parser.TreeParser.ErrorListener;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StepColumn;
import ru.yandex.solomon.model.point.column.TsColumn;
import ru.yandex.solomon.model.point.column.ValueColumn;
import ru.yandex.solomon.model.timeseries.AggrGraphDataArrayList;
import ru.yandex.solomon.proto.UrlStatusType;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.api.EDecimPolicy;

import static ru.yandex.solomon.coremon.stockpile.Validator.isNotValid;


/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class MetricProcessorImpl implements MetricProcessor {
    // previously coremon write only DGauge
    private static final ru.yandex.solomon.model.protobuf.MetricType WRITE_TYPE = ru.yandex.solomon.model.protobuf.MetricType.DGAUGE;
    private static final int COLUMN_SET = TsColumn.mask | ValueColumn.mask | StepColumn.mask;

    // bunch of helpful helpers
    private final AggrHelper aggrHelper;
    private final CoremonShardStockpileResolveHelper resolveHelper;
    private final CoremonShardStockpileWriteHelper writeHelper;
    private final MemOnlyMetricsCollection memOnlyMetrics;

    // additional incoming data
    private final Labels optLabelSorted;
    private final boolean isPush;
    private final long responseTsMillis;
    private final long gridMillis;
    private final long responseDiffMillis;
    private final long fetchIntervalMillis;
    private final Supplier<MetricsPrevValues> prevValuesSupplier;
    @Nullable
    private MetricsPrevValues prevValues;

    // shard configs
    private final EDecimPolicy decimPolicyId;
    private final CoremonShardQuota quota;
    private final boolean isShardMemOnly;
    private final ValidationMode validationMode;
    private final boolean gridRounding;

    // stats
    private final ErrorListener errors;
    private final MetricStats metricStats = new MetricStats();

    // mutable state
    private final AggrPoint tempPoint = new AggrPoint();
    private final AggrGraphDataArrayList tempTimeseries = new AggrGraphDataArrayList(COLUMN_SET, 0);
    private final MemOnlyCoremonMetric memOnlyMetric = new MemOnlyCoremonMetric();
    @Nullable
    private CoremonMetric metric;
    private Labels labels;
    private MetricType type;
    private Status status = Status.OK;

    public MetricProcessorImpl(
            AggrHelper aggrHelper,
            CoremonShardStockpileResolveHelper resolveHelper,
            CoremonShardStockpileWriteHelper writeHelper,
            MemOnlyMetricsCollection memOnlyMetrics,
            Labels optLabelSorted,
            boolean isPush,
            long responseTsMillis,
            long prevResponseTsMillis,
            Supplier<MetricsPrevValues> prevValues,
            long configuredGridMillis,
            long configuredFetchMillis,
            EDecimPolicy decimPolicyId,
            CoremonShardQuota quota,
            boolean isShardMemOnly,
            ErrorListener errors,
            ValidationMode validationMode,
            boolean gridRounding)
    {
        this.aggrHelper = aggrHelper;
        this.resolveHelper = resolveHelper;
        this.writeHelper = writeHelper;
        this.memOnlyMetrics = memOnlyMetrics;
        this.optLabelSorted = optLabelSorted;
        this.prevValuesSupplier = prevValues;
        this.decimPolicyId = decimPolicyId;
        this.quota = quota;
        this.gridRounding = gridRounding;
        this.isShardMemOnly = isShardMemOnly;
        this.isPush = isPush;
        this.gridMillis = configuredGridMillis;
        this.responseTsMillis = truncate(responseTsMillis);
        this.responseDiffMillis = prevResponseTsMillis > 0
                ? responseTsMillis - prevResponseTsMillis
                : configuredFetchMillis;
        this.fetchIntervalMillis = configuredFetchMillis;
        this.errors = errors;
        this.validationMode = validationMode;

        if (responseTsMillis == 0) {
            throw new IllegalArgumentException("response timestamp is not specified");
        }
    }

    private static void checkTs(Labels labels, long tsMillis) {
        if (!InstantUtils.isGoodMillis(tsMillis)) {
            throw new UrlStatusTypeException(
                    "invalid timestamp " + InstantUtils.formatToSeconds(tsMillis) + " for metric " + labels,
                    UrlStatusType.PARSE_ERROR);
        }
    }

    private boolean isDeriv(MetricType type) {
        return type == MetricType.RATE || type == MetricType.HIST_RATE;
    }

    @Override
    public void onMetricBegin(MetricType type, Labels labels, boolean memOnly) {
        if (metric != null) {
            metric.close();
            metric = null;
        }

        final int count = metricStats.incCount(type);
        if (count > quota.getMaxMetricsPerUrl()) {
            String message = "more than " + quota.getMaxMetricsPerUrl() + " metrics from one URL";
            throw new UrlStatusTypeException(message, UrlStatusType.QUOTA_ERROR);
        }

        this.labels = labels;
        this.type = type;


        if (memOnly || isShardMemOnly) {
            var aggrs = memOnlyMetrics.getOrNull(optLabelSorted, labels);
            memOnlyMetric.setLabels(labels);
            memOnlyMetric.setAggrMetrics(aggrs);

            if (aggrs == null) {
                metricStats.incUnknown();
                if (isNotValid(validationMode, labels, errors)) {
                    return;
                }
                if (memOnlyMetrics.size() > quota.getMaxMemMetrics()) {
                    String message = "more than " + quota.getMaxMemMetrics() + " memonly metrics in shard";
                    status = new Status(UrlStatusType.QUOTA_ERROR, message);
                    return;
                }
            }
            metric = memOnlyMetric;
        } else {
            CoremonMetric fileMetric = resolveHelper.tryResolveMetric(labels, type);
            if (fileMetric == null) {
                metricStats.incUnknown();
                if (isNotValid(validationMode, labels, errors)) {
                    return;
                }
                if (resolveHelper.size() > quota.getMaxFileMetrics()) {
                    String message = "more than " + quota.getMaxFileMetrics() + " metrics in shard";
                    status = new Status(UrlStatusType.QUOTA_ERROR, message);
                    return;
                }
                // TODO: do not allocate this temporary metric object to process aggr requests
                fileMetric = new FileCoremonMetric(0, 0, labels, type);
            }
            metric = fileMetric;
        }
    }

    @Override
    public void onPoint(long tsMillis, double value) {
        if (metric == null) {
            return;
        }
        final boolean deriv = isDeriv(type);
        if (deriv) {
            if (tsMillis != 0 || isPush) {
                throw new UrlStatusTypeException(UrlStatusType.DERIV_AND_TS);
            }
        }

        tempPoint.columnSet = COLUMN_SET;
        tempPoint.tsMillis = actualizeTsMillis(metric, tsMillis);

        if (deriv) {
            tempPoint.valueNum = value - prevValue(labels);
            if (tempPoint.valueNum < 0) {
                return;
            }

            tempPoint.stepMillis = responseDiffMillis;
            tempPoint.valueDenom = responseDiffMillis;
        } else {
            tempPoint.valueNum = value;
            tempPoint.valueDenom = ValueColumn.DEFAULT_DENOM;
            tempPoint.stepMillis = gridMillis;
        }

        if (Double.isNaN(tempPoint.valueNum)) {
            // fast pass for ignoring NaN values
            return;
        }

        onPoint(metric, tempPoint);
    }

    private void onPoint(CoremonMetric metric, AggrPoint point) {
        if (!metric.isMemOnly()) {
            if (metric.getLocalId() == 0) {
                // TODO: after resolve write as a DGauge, not as a type value (gordiychuk@)
                resolveHelper.addMetricData(new UnresolvedMetricPoint(labels, type, WRITE_TYPE, point));
            } else {
                metric.setLastPointSeconds(InstantUtils.millisecondsToSeconds(point.tsMillis));
                writePoint(metric, point);
            }
        }

        writeAggregatePoint(metric, point);
    }

    @Override
    public void onPoint(long tsMillis, long value) {
        onPoint(tsMillis, (double) value);
    }

    @Override
    public void onTimeSeries(TimeSeries timeSeries) {
        if (metric == null) {
            return;
        }
        if (isDeriv(type)) {
            // TODO: support deriv metric with timeseries
            throw new UrlStatusTypeException(UrlStatusType.DERIV_AND_TS);
        }

        metricStats.incTimeseries();
        if (timeSeries.size() == 0) {
            return;
        } else if (timeSeries.size() == 1) {
            if (timeSeries.isDouble()) {
                onPoint(timeSeries.tsMillisAt(0), timeSeries.doubleAt(0));
            } else {
                onPoint(timeSeries.tsMillisAt(0), timeSeries.longAt(0));
            }
            return;
        }

        metricStats.incTimeseriesMoreThenOnePoint();
        onTimeseries(metric, toList(timeSeries));
    }

    private void onTimeseries(CoremonMetric metric, AggrGraphDataArrayList timeSeries) {
        if (timeSeries.isEmpty()) {
            return;
        }

        if (!metric.isMemOnly()) {
            if (metric.getLocalId() == 0) {
                var metricTimeSeries = new UnresolvedMetricTimeSeries(labels, type, WRITE_TYPE, timeSeries.clone());
                resolveHelper.addMetricData(metricTimeSeries);
            } else {
                final long lastTsMillis = timeSeries.getTsMillis(timeSeries.length() - 1);
                metric.setLastPointSeconds(InstantUtils.millisecondsToSeconds(lastTsMillis));
                writeTimeseries(metric, timeSeries);
            }
        }

        // add one aggregated point per fetch interval interval no grid millis,
        // it's important, see https://st.yandex-team.ru/SOLOMON-5506#5ece8be1802a026dec1fc5ee
        //
        // use configured fetch interval, not real distance between two responses
        // because slow down on fetcher should not affect to change grid in aggregates
        // and as a result apply weighted avg where it is not necessary,
        // see https://st.yandex-team.ru/SOLOMON-7423#611d3d8a6f970a5e0e804dc1
        writeAggregateTimeSeries(metric, type, labels, fetchIntervalMillis, timeSeries);
    }

    private AggrGraphDataArrayList toList(TimeSeries timeSeries) {
        tempTimeseries.truncate(0);
        tempTimeseries.ensureCapacity(COLUMN_SET, timeSeries.size());
        for (int i = 0; i < timeSeries.size(); i++) {
            var tsMillis = truncate(timeSeries.tsMillisAt(i));
            checkTs(labels, tsMillis);

            var value = timeSeries.isDouble()
                    ? timeSeries.doubleAt(i)
                    : (double) timeSeries.longAt(i);

            if (Double.isNaN(value)) {
                continue;
            }

            Points.fillSimple(tempPoint,
                    tsMillis,
                    gridMillis,
                    value,
                    ValueColumn.DEFAULT_DENOM);
            tempTimeseries.addRecord(tempPoint);
        }
        return tempTimeseries;
    }

    private double prevValue(Labels labels) {
        if (prevValues == null) {
            prevValues = prevValuesSupplier.get();
        }

        return prevValues.getDouble(labels);
    }

    private void writePoint(CoremonMetric metric, AggrPoint point) {
        writeHelper.addPoint(metric.getShardId(), metric.getLocalId(), point, decimPolicyId.getNumber(), WRITE_TYPE);
    }

    private void writeAggregatePoint(CoremonMetric metric, AggrPoint point) {
        if (!aggrHelper.hasAggrRules() || AggrMetrics.isEmpty(metric.getAggrMetrics())) {
            return;
        }

        point.setMerge(true);
        point.setCount(1);
        if (point.stepMillis != StepColumn.DEFAULT_VALUE) {
            point.tsMillis = InstantUtils.truncate(point.tsMillis, point.stepMillis);
        }

        boolean updated = aggrHelper.aggregatePoint(
                resolveHelper,
                writeHelper,
                labels,
                point,
                type,
                WRITE_TYPE,
                metric,
                optLabelSorted);

        if (updated && metric.isMemOnly()) {
            Object aggrMetrics = metric.getAggrMetrics();
            assert aggrMetrics != null;
            memOnlyMetrics.put(optLabelSorted, metric.getLabels(), aggrMetrics);
        }
    }

    private void writeTimeseries(CoremonMetric metric, AggrGraphDataArrayList timeSeries) {
        for (int i = 0; i < timeSeries.length(); i++) {
            timeSeries.getDataTo(i, tempPoint);
            tempPoint.columnSet = COLUMN_SET;
            writePoint(metric, tempPoint);
        }
    }

    private void writeAggregateTimeSeries(CoremonMetric metric, MetricType type, Labels labels, long gridMillis, AggrGraphDataArrayList timeSeries) {
        if (!aggrHelper.hasAggrRules() || AggrMetrics.isEmpty(metric.getAggrMetrics())) {
            return;
        }

        boolean updated = aggrHelper.aggregateTimeseries(
                resolveHelper,
                writeHelper,
                labels,
                type,
                tempPoint,
                timeSeries,
                gridMillis,
                metric, optLabelSorted);

        if (updated && metric.isMemOnly()) {
            Object aggrMetric = metric.getAggrMetrics();
            assert aggrMetric != null;
            memOnlyMetrics.put(optLabelSorted, metric.getLabels(), aggrMetric);
        }
    }

    private long actualizeTsMillis(CoremonMetric metric, long tsMillis) {
        if (tsMillis == 0) {
            return responseTsMillis;
        } else {
            checkTs(metric.getLabels(), tsMillis);
            return truncate(tsMillis);
        }
    }

    private long truncate(long tsMillis) {
        if (!gridRounding) {
            return tsMillis;
        }

        return InstantUtils.truncate(tsMillis, gridMillis);
    }

    @Override
    public MetricStats getMetricStats() {
        return metricStats;
    }

    @Override
    public Status getStatus() {
        return status;
    }

    @Override
    public void close() {
        if (metric != null) {
            metric.close();
            metric = null;
        }
    }
}
