package ru.yandex.solomon.coremon.stockpile;

import java.lang.management.ManagementFactory;
import java.lang.management.ThreadMXBean;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;

import ru.yandex.misc.actor.ActorRunnerImpl;
import ru.yandex.misc.actor.Tasks;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.lang.Validate;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.solomon.util.ExceptionUtils;

/**
 * @author Stepan Koltsov
 *
 * @see ru.yandex.misc.actor.ActorWithFutureRunner
 */
public class ParsingActorRunner {

    private static final ThreadMXBean threadMXBean = ManagementFactory.getThreadMXBean();
    static {
        threadMXBean.setThreadCpuTimeEnabled(true);
    }
    /** Scheduling state */
    final Tasks tasks = new Tasks();
    /** Helper runnable */
    private final RunnableImpl runnable = new RunnableImpl();

    private final AtomicInteger waitingFutures = new AtomicInteger();
    private final int waitingFuturesLimit;
    private final Meter cpuTimeNanos;

    public interface ActorBody {
        CompletableFuture<?> run();
    }

    private final ActorBody actor;
    private final Executor executor;

    ParsingActorRunner(
        int waitingFuturesLimit,
        ActorBody actor,
        Executor executor,
        Meter cpuTimeNanos)
    {
        Validate.isTrue(waitingFuturesLimit > 0);

        this.waitingFuturesLimit = waitingFuturesLimit;
        this.actor = actor;
        this.executor = executor;
        this.cpuTimeNanos = cpuTimeNanos;
    }

    /**
     * Schedule actor.
     *
     * If actor is sleeping, then actor will be executed right now.
     * If actor is executing right now, it will be executed one more time.
     * If this method is called multiple time, actor will be re-executed no more than one more time.
     */
    public void schedule() {
        if (tasks.addTask()) {
            ActorRunnerImpl.schedule(executor, runnable);
        }
    }

    private void loop() {
        while (tasks.fetchTask()) {
            var startNanos = threadMXBean.getCurrentThreadCpuTime();
            try {
                if (!iteration()) {
                    break;
                }
            } finally {
                var stopNanos = threadMXBean.getCurrentThreadCpuTime();
                cpuTimeNanos.mark(stopNanos - startNanos);
            }
        }
    }

    private boolean iteration() {
        var future = actor.run();

        // If actor body already completed a future
        // it is cheaper to execute it here instead of scheduling.
        //
        // Returning completed future is equivalent to using plain `ActorRunner`
        if (CompletableFutures.isCompletedSuccessfully(future)) {
            return true;
        }

        var continueLoop = waitingFutures.incrementAndGet() < waitingFuturesLimit;
        future.whenComplete((r, thr) -> {
            if (thr != null) {
                ExceptionUtils.uncaughtException(thr);
            }

            if (waitingFutures.getAndDecrement() == waitingFuturesLimit) {
                try {
                    if (tasks.checkTask()) {
                        ActorRunnerImpl.schedule(executor, runnable);
                    }
                } catch (Throwable t) {
                    ExceptionUtils.uncaughtException(t);
                }
            }
        });
        return continueLoop;
    }

    private class RunnableImpl implements Runnable {
        @Override
        public void run() {
            try {
                loop();
            } catch (Throwable e) {
                ExceptionUtils.uncaughtException(e);
            }
        }
    }
}
