package ru.yandex.solomon.coremon.stockpile;

import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;

import org.apache.logging.log4j.util.Strings;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.coremon.CoremonState;
import ru.yandex.solomon.model.protobuf.MetricFormat;

/**
 * @author Vladimir Gordiychuk
 */
public class ServiceProviderMetricsSupplier implements MetricSupplier {

    private final CoremonState state;
    private volatile int estimatedCount = 0;

    public ServiceProviderMetricsSupplier(CoremonState state) {
        this.state = state;
    }

    @Override
    public int estimateCount() {
        return estimatedCount;
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        var it = state.getInitializedShards().iterator();
        var byServiceProvider = new HashMap<String, ServiceProviderMetrics>();
        while (it.hasNext()) {
            var shard = it.next();
            var processingShard = shard.getProcessingShard();

            var serviceProvider = processingShard.getOpts().getServiceProvider();
            if (Strings.isEmpty(serviceProvider)) {
                continue;
            }

            var metrics = byServiceProvider.get(serviceProvider);
            if (metrics == null) {
                metrics = new ServiceProviderMetrics(serviceProvider);
                byServiceProvider.put(serviceProvider, metrics);
            }

            metrics.add(processingShard.getMetrics());
        }

        if (byServiceProvider.isEmpty()) {
            estimatedCount = 0;
            return;
        }

        int count = 0;
        var total = new ServiceProviderMetrics("total");
        for (var metrics : byServiceProvider.values()) {
            count += metrics.estimateCount();
            metrics.append(tsMillis, commonLabels, consumer);
            total.combine(metrics);
        }

        count += total.estimateCount();
        total.append(tsMillis, commonLabels, consumer);
        estimatedCount = count;
    }

    static class ServiceProviderMetrics implements MetricSupplier {
        final MetricRegistry registry;
        final GaugeInt64 shardsCount;
        final Meter cpuTimeNanos;

        final Map<MetricFormat, Rate> inboundDocuments = new EnumMap<>(MetricFormat.class);

        final Meter inboundBytes;
        final Rate outboundBytes;

        final Rate inboundMetrics;
        final Rate outboundPoints;

        final GaugeInt64 fileMetrics;
        final GaugeInt64 inMemoryMetrics;

        final Histogram fileMetricsInShard;
        final Histogram inMemoryMetricsInShard;

        final GaugeInt64 documentsQueueSize;
        final Rate documentsNotMatchInterval;

        public ServiceProviderMetrics(String serviceProvider) {
            this.registry = new MetricRegistry(Labels.of("service_provider", serviceProvider));
            this.shardsCount = registry.gaugeInt64("metabase.shards_count");
            this.cpuTimeNanos = registry.fiveMinutesMeter("processing.cpu_time_nanos");

            this.inboundBytes = registry.fiveMinutesMeter("processing.inbound_bytes");
            this.outboundBytes = registry.rate("processing.outbound_bytes");

            this.inboundMetrics = registry.rate("processing.inbound_metrics");

            this.outboundPoints = registry.rate("processing.outbound_points");

            this.fileMetrics = registry.gaugeInt64("metabase.file_metrics");
            this.inMemoryMetrics = registry.gaugeInt64("metabase.in_memory_metrics");

            this.fileMetricsInShard = registry.histogramCounter("metabase.shard.file_metrics", Histograms.exponential(30, 2));
            this.inMemoryMetricsInShard = registry.histogramCounter("metabase.shard.in_memory_metrics", Histograms.exponential(30, 2));

            this.documentsQueueSize = registry.gaugeInt64("processing.documents.queue_size");
            this.documentsNotMatchInterval = registry.rate("processing.documents.not_match_interval");
        }

        public void add(CoremonShardStockpileMetrics metrics) {
            shardsCount.add(1);
            cpuTimeNanos.combine(metrics.cpuTimeNanos);

            for (var entry : metrics.docByFormat.entrySet()) {
                combine(entry.getKey(), entry.getValue());
            }

            inboundBytes.combine(metrics.bytesReceived);
            outboundBytes.combine(metrics.bytesWritten);

            inboundMetrics.combine(metrics.metricsTotal);

            outboundPoints.combine(metrics.pushMetricsToStorage);

            fileMetrics.combine(metrics.fileMetrics);
            fileMetricsInShard.record(metrics.fileMetrics.get());

            inMemoryMetrics.combine(metrics.inMemMetrics);
            inMemoryMetricsInShard.record(metrics.inMemMetrics.get());

            documentsQueueSize.combine(metrics.queueSize);
            documentsNotMatchInterval.combine(metrics.documentsNotMatchInterval);
        }

        public void combine(MetricFormat format, Rate inbound) {
            var target = inboundDocuments.get(format);
            if (target == null) {
                target = registry.rate("processing.inbound_documents", Labels.of("format", format.name()));
                inboundDocuments.put(format, target);
            }
            target.combine(inbound);
        }

        public void combine(ServiceProviderMetrics metrics) {
            this.shardsCount.combine(metrics.shardsCount);
            this.cpuTimeNanos.combine(metrics.cpuTimeNanos);

            for (var entry : metrics.inboundDocuments.entrySet()) {
                combine(entry.getKey(), entry.getValue());
            }

            this.inboundBytes.combine(metrics.inboundBytes);
            this.outboundBytes.combine(metrics.outboundBytes);

            this.inboundMetrics.combine(metrics.inboundMetrics);

            this.outboundPoints.combine(metrics.outboundPoints);

            this.fileMetrics.combine(metrics.fileMetrics);
            this.fileMetricsInShard.combine(metrics.fileMetricsInShard);

            this.inMemoryMetrics.combine(metrics.inMemoryMetrics);
            this.inMemoryMetricsInShard.combine(metrics.inMemoryMetricsInShard);

            this.documentsQueueSize.combine(metrics.documentsQueueSize);
            this.documentsNotMatchInterval.combine(metrics.documentsNotMatchInterval);
        }

        @Override
        public int estimateCount() {
            return registry.estimateCount();
        }

        @Override
        public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
            registry.append(tsMillis, commonLabels, consumer);
        }
    }
}
