package ru.yandex.solomon.coremon.stockpile;

import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import io.netty.buffer.ByteBuf;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.ProcessingResult;
import ru.yandex.solomon.memory.layout.MemMeasurable;
import ru.yandex.solomon.memory.layout.MemoryCounter;
import ru.yandex.solomon.model.protobuf.MetricFormat;
import ru.yandex.solomon.staffOnly.manager.special.InstantMillis;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class TwoResponsesRequest extends CompletableFuture<ProcessingResult> implements MemMeasurable {
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(TwoResponsesRequest.class);

    public final Labels commonLabels;
    public final Labels commonOptLabels;
    public final MetricFormat metricFormat;
    @Nullable
    public ByteBuf request;
    @InstantMillis
    public final long requestInstant;
    @Nullable
    public ByteBuf prevRequest;
    @InstantMillis
    public final long prevInstant;
    public final boolean isPush;
    public final boolean onlyNewFormatWrites;
    public ProcessingResult delayedResult;

    public TwoResponsesRequest(
        Labels commonLabels, Labels commonOptLabels,
        MetricFormat metricFormat,
        ByteBuf request, long requestInstant,
        ByteBuf prevRequest, long prevInstant,
        boolean isPush, boolean onlyNewFormatWrites)
    {
        this.metricFormat = metricFormat;
        this.isPush = isPush;
        this.onlyNewFormatWrites = onlyNewFormatWrites;
        if (prevInstant >= requestInstant) {
            throw new IllegalArgumentException(String.format("prevInstant %d is >= than the requestInstant %d", prevInstant, requestInstant));
        }

        this.commonLabels = commonLabels;
        this.commonOptLabels = commonOptLabels;
        this.request = request;
        this.requestInstant = requestInstant;
        this.prevRequest = prevRequest;
        this.prevInstant = prevInstant;
    }

    /** Release memory */
    public void release() {
        if (request != null) {
            request.release();
            request = null;
        }
        if (prevRequest != null) {
            prevRequest.release();
            prevRequest = null;
        }
    }

    @Override
    public long memorySizeIncludingSelf() {
        // self size of ByteBuf is ignored, because it depends on actual implementation
        return SELF_SIZE +
            (request == null ? 0 : request.capacity()) +
            (prevRequest == null ? 0 : prevRequest.capacity());
    }
}
