package ru.yandex.solomon.coremon.stockpile.write;

import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;

import javax.annotation.ParametersAreNonnullByDefault;

import io.netty.util.internal.RecyclableArrayList;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.coremon.stockpile.CoremonShardStockpileMetrics;
import ru.yandex.solomon.coremon.stockpile.CoremonShardStockpileMetrics.WriteToStorageResult;
import ru.yandex.solomon.coremon.stockpile.CoremonShardStockpileWriteHelper;
import ru.yandex.solomon.coremon.stockpile.ProcessingWriter;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.point.column.StockpileColumnSet;
import ru.yandex.solomon.model.point.column.StockpileColumns;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.staffOnly.annotations.LinkedOnRootPage;
import ru.yandex.stockpile.api.EProjectId;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.writeRequest.StockpileShardWriteRequest;
import ru.yandex.stockpile.client.writeRequest.StockpileShardWriteRequestBuilder;

/**
 * @author Sergey Polovko
 */
@LinkedOnRootPage("Stockpile Buffered Writer")
public class StockpileBufferedWriter implements ProcessingWriter {

    /* package */ static final long MAX_BATCH_SIZE_IN_BYTES = 128 << 20; // 128 MiB

    private final StockpileClient stockpileClient;
    private final Executor executor;
    private volatile StockpileWriteQueue[] queueByShardId;

    public StockpileBufferedWriter(StockpileClient stockpileClient, Executor executor, MetricRegistry metricRegistry) {
        this.stockpileClient = stockpileClient;
        this.executor = executor;
        this.queueByShardId = changeQueueSize(new StockpileWriteQueue[0], stockpileClient.getTotalShardsCount());

        metricRegistry.lazyGaugeInt64("stockpile.writeQueueSize", () -> {
            long size = 0;
            for (var queue : queueByShardId) {
                size += queue.size();
            }
            return size;
        });

        metricRegistry.lazyGaugeInt64("stockpile.writeQueueBytes", () -> {
            long bytes = 0;
            for (var queue : queueByShardId) {
                bytes += queue.bytes();
            }
            return bytes;
        });
    }

    private StockpileWriteQueue queue(int shardId) {
        var copy = queueByShardId;
        if (shardId > copy.length) {
            int totalShards = Math.max(shardId, stockpileClient.getTotalShardsCount());
            if (copy.length < totalShards) {
                synchronized (this) {
                    copy = changeQueueSize(queueByShardId, totalShards);
                    queueByShardId = copy;
                }
            } else {
                throw new IllegalArgumentException("unknown shard id: " + shardId);
            }
        }
        return copy[shardId - 1];
    }

    private StockpileWriteQueue[] changeQueueSize(StockpileWriteQueue[] prev, int expectedSize) {
        if (prev.length >= expectedSize) {
            return prev;
        }

        var result = Arrays.copyOf(prev, expectedSize);
        for (int index = prev.length; index < result.length; index++) {
            int shardId = index + 1;
            result[index] = new StockpileWriteQueue(shardId, stockpileClient, executor);
        }
        return result;
    }

    public CoremonShardStockpileWriteHelper newHelper(int coremonShardId, CoremonShardStockpileMetrics metrics) {
        return new WriteHelper(coremonShardId, metrics);
    }

    @Override
    public String toString() {
        return "StockpileBufferedWriter{" +
            ", batchSize=" + DataSize.shortString(MAX_BATCH_SIZE_IN_BYTES) +
            '}';
    }

    /**
     * WRITE HELPER
     */
    @ParametersAreNonnullByDefault
    private final class WriteHelper implements CoremonShardStockpileWriteHelper {

        private final int coremonShardId;
        private final CoremonShardStockpileMetrics metrics;
        private final Int2ObjectOpenHashMap<StockpileShardWriteRequestBuilder> writesByShard;
        private int pointCount = 0;

        WriteHelper(int coremonShardId, CoremonShardStockpileMetrics metrics) {
            this.coremonShardId = coremonShardId;
            this.metrics = metrics;
            this.writesByShard = new Int2ObjectOpenHashMap<>(32);
        }

        @Override
        public void addPoint(int shardId, long localId, AggrPoint point, int decimPolicyId, MetricType type) {
            if (point.columnSetMask() == StockpileColumnSet.empty.columnSetMask()) {
                return;
            }

            StockpileColumns.ensureColumnSetValid(type, point.columnSet);
            var builder = writesByShard.get(shardId);
            if (builder == null) {
                builder = newWriteBuilder();
                writesByShard.put(shardId, builder);
            }
            builder.addRecord(localId, point, decimPolicyId, type);
            pointCount++;
        }

        private StockpileShardWriteRequestBuilder newWriteBuilder() {
            return new StockpileShardWriteRequestBuilder(EProjectId.SOLOMON, coremonShardId);
        }

        @Override
        public CompletableFuture<Void> write() {
            if (writesByShard.isEmpty()) {
                return CompletableFuture.completedFuture(null);
            }

            var futures = RecyclableArrayList.newInstance(writesByShard.size());
            try {
                var it = writesByShard.int2ObjectEntrySet().fastIterator();
                while (it.hasNext()) {
                    var entry = it.next();
                    int shardId = entry.getIntKey();
                    var builder = entry.getValue();
                    var request = builder.build();
                    futures.add(writeToShard(shardId, request));
                    it.remove();
                }
                metrics.addPushMetricsToStorage(pointCount);
                return CompletableFuture.allOf(futures.toArray(new CompletableFuture[0]));
            } finally {
                futures.recycle();
            }
        }

        private CompletableFuture<EStockpileStatusCode> writeToShard(int shardId, StockpileShardWriteRequest request) {
            final long records = request.countRecords();
            final long bytes = request.getSizeBytes();

            return queue(shardId)
                    .enqueue(request)
                    .whenComplete((status, throwable) -> {
                        WriteToStorageResult result = (throwable != null || status != EStockpileStatusCode.OK)
                                ? WriteToStorageResult.ERROR
                                : WriteToStorageResult.SUCCESS;
                        metrics.addMetricProcessResults(result, records, bytes);
                    });
        }
    }
}
