package ru.yandex.solomon.coremon.stockpile.write;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.UnsafeByteOperations;
import io.netty.util.internal.RecyclableArrayList;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.TWriteDataBinaryRequest;
import ru.yandex.stockpile.client.StockpileClient;
import ru.yandex.stockpile.client.writeRequest.StockpileShardWriteRequest;

import static org.apache.commons.lang3.ArrayUtils.EMPTY_BYTE_ARRAY;
import static ru.yandex.solomon.coremon.stockpile.write.StockpileBufferedWriter.MAX_BATCH_SIZE_IN_BYTES;

/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
final class StockpileWriteQueue implements ActorWithFutureRunner.ActorBody {

    private static final AtomicLongFieldUpdater<StockpileWriteQueue> queueSizeUpdater =
        AtomicLongFieldUpdater.newUpdater(StockpileWriteQueue.class, "queueSize");

    private static final AtomicLongFieldUpdater<StockpileWriteQueue> queueBytesUpdater =
        AtomicLongFieldUpdater.newUpdater(StockpileWriteQueue.class, "queueBytes");

    private final int shardId;
    private final StockpileClient stockpileClient;
    private final ActorWithFutureRunner actor;
    private final ConcurrentLinkedQueue<Item> queue = new ConcurrentLinkedQueue<>();

    private volatile long queueSize = 0;
    private volatile long queueBytes = 0;

    StockpileWriteQueue(int shardId, StockpileClient stockpileClient, Executor executor) {
        this.shardId = shardId;
        this.stockpileClient = stockpileClient;
        this.actor = new ActorWithFutureRunner(this, executor);
    }

    CompletableFuture<EStockpileStatusCode> enqueue(StockpileShardWriteRequest request) {
        queueSizeUpdater.incrementAndGet(this);
        queueBytesUpdater.addAndGet(this, request.getSizeBytes());

        Item item = new Item(request);
        queue.offer(item);
        actor.schedule();
        return item;
    }

    @Override
    public CompletableFuture<?> run() {
        var future = write(dequeRequests());
        while (queueBytes >= MAX_BATCH_SIZE_IN_BYTES) {
            future = write(dequeRequests());
        }
        return future;
    }

    private CompletableFuture<?> write(SerializedRequests requests) {
        if (requests == SerializedRequests.EMPTY) {
            return CompletableFuture.completedFuture(null);
        }

        var request = TWriteDataBinaryRequest.newBuilder()
            .setShardId(shardId)
            .setContent(UnsafeByteOperations.unsafeWrap(requests.content))
            .build();

        requests.content = null;
        Item[] items = requests.items;

        return stockpileClient.writeDataBinary(request)
            .whenComplete((response, e) -> {
                if (e != null) {
                    for (Item item : items) {
                        item.completeExceptionally(e);
                    }
                } else {
                    for (Item item : items) {
                        item.complete(response.getStatus());
                    }
                }
            });
    }

    private SerializedRequests dequeRequests() {
        final RecyclableArrayList requests = RecyclableArrayList.newInstance();
        final RecyclableArrayList items = RecyclableArrayList.newInstance();
        try {
            long batchSize = 0;
            Item item;
            while (batchSize < MAX_BATCH_SIZE_IN_BYTES && (item = queue.poll()) != null) {
                batchSize += item.getSizeBytes(); // actual batch size can be above the limit by size of the last item
                requests.add(item.takeRequest());
                items.add(item);
            }

            if (requests.isEmpty()) {
                return SerializedRequests.EMPTY;
            }

            queueSizeUpdater.addAndGet(this, -1 * requests.size());
            queueBytesUpdater.addAndGet(this, -1 * batchSize);

            @SuppressWarnings("unchecked")
            byte[] content = StockpileShardWriteRequest.serialize((Iterable) requests);
            for (Object request : requests) {
                ((StockpileShardWriteRequest) request).close();
            }

            @SuppressWarnings("SuspiciousToArrayCall")
            Item[] itemsArray = items.toArray(Item[]::new);
            return new SerializedRequests(content, itemsArray);
        } finally {
            requests.recycle();
            items.recycle();
        }
    }

    public long size() {
        return queueSizeUpdater.get(this);
    }

    public long bytes() {
        return queueBytesUpdater.get(this);
    }

    /**
     * SERIALIZED ITEMS
     */
    private static final class SerializedRequests {
        static final SerializedRequests EMPTY = new SerializedRequests(EMPTY_BYTE_ARRAY, new Item[0]);

        private byte[] content;
        private Item[] items;

        SerializedRequests(byte[] content, Item[] items) {
            this.content = content;
            this.items = items;
        }
    }

    /**
     * QUEUE ITEM
     *
     * XXX: extends CompletableFuture to reduce allocations
     */
    @ParametersAreNonnullByDefault
    private static final class Item extends CompletableFuture<EStockpileStatusCode> {
        @Nullable
        private StockpileShardWriteRequest request;

        Item(StockpileShardWriteRequest request) {
            this.request = request;
        }

        long getSizeBytes() {
            assert request != null;
            return request.getSizeBytes();
        }

        StockpileShardWriteRequest takeRequest() {
            StockpileShardWriteRequest request = this.request;
            this.request = null;
            return request;
        }
    }
}
