package ru.yandex.solomon.coremon.tasks;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Message;
import com.google.protobuf.MessageOrBuilder;
import com.google.protobuf.TextFormat;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.future.RetryConfig;

import static java.lang.System.currentTimeMillis;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.util.time.DurationUtils.backoff;
import static ru.yandex.solomon.util.time.DurationUtils.randomize;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
public abstract class AbstractTask<P extends Message> implements AutoCloseable {

    private static final long BACKOFF_DELAY_MILLIS = TimeUnit.MINUTES.toMillis(10);
    private static final long BACKOFF_MAX_DELAY_MILLIS = TimeUnit.DAYS.toMillis(1);

    private final Logger logger = LoggerFactory.getLogger(getClass());

    private final String type;

    private final RetryConfig retryConfig;

    private final ExecutionContext context;
    private final MessageOrBuilder params;
    private final AtomicReference<P> progress;

    private final PingActorRunner actor;

    protected AbstractTask(
        String type,
        RetryConfig retryConfig,
        ExecutionContext context,
        MessageOrBuilder params,
        P progress,
        Executor executor,
        ScheduledExecutorService timer)
    {
        this.type = type;

        this.retryConfig = retryConfig
            .withStats((timeSpentMillis, cause) -> logger.warn("{} error", logPrefix(), cause));

        this.context = context;
        this.params = params;
        this.progress = new AtomicReference<>(progress);

        this.actor = PingActorRunner.newBuilder()
            .pingInterval(Duration.ofSeconds(15))
            .operation(type + "_save_progress")
            .executor(executor)
            .timer(timer)
            .onPing(this::saveProgress)
            .build();
    }

    public final CompletableFuture<Void> start() {
        actor.schedule();
        var root = new CompletableFuture<>();
        var future = root.thenCompose(i -> onStart())
            .handle((i, t) -> t != null ? Status.fromThrowable(t) : Status.OK)
            .thenCompose(status -> {
                if (context.isDone()) {
                    return completedFuture(null);
                }

                if (status.isOk()) {
                    return complete(status);
                }

                return reschedule(status);
            })
            .whenComplete((i, t) -> close(t));

        root.complete(null);
        return future;
    }

    @Override
    public final void close() {
        if (logger.isDebugEnabled()) {
            logger.debug("{} close", logPrefix());
        }

        if (!context.isDone()) {
            context.cancel();
        }

        onClose();

        actor.close();
    }

    protected abstract CompletionStage<Void> onStart();

    protected abstract P onUpdateProgress(P prev, @Nullable Status status);

    protected abstract Message result(P progress);

    protected abstract int attempt(P progress);

    protected abstract void onClose();

    protected void afterReschedule(long executeAt, P progress, @Nullable Status status) { /* no-op */ }

    protected void afterComplete(Message result) { /* no-op */ }

    protected final RetryConfig retryConfig() {
        return retryConfig;
    }

    protected final CompletableFuture<Void> forceSave() {
        return actor.forcePing();
    }

    protected final CompletableFuture<Void> reschedule(long executeAt) {
        return reschedule(executeAt, progress(), null);
    }

    //TODO: do it within a single tx in dao
    protected final CompletableFuture<Void> fail(Status status) {
        return context.progress(progress())
            .thenCompose(i -> doFail(status));
    }


    private CompletableFuture<Void> doFail(Status status) {
        actor.close();
        return context.fail(new StatusRuntimeExceptionNoStackTrace(status))
            .thenRun(() -> logger.info(
                "{} fail {}, latest progress ({})",
                logPrefix(), status, TextFormat.shortDebugString(progress.get())));
    }

    private CompletionStage<Void> complete(Status status) {
        var progress = progress(status);
        var result = result(progress);

        return context.progress(progress)
            .thenCompose(i -> complete(result));
    }

    //TODO: do it within a single tx in dao
    private CompletableFuture<Void> complete(Message result) {
        actor.close();
        return context.complete(result)
            .thenRun(() -> {
                logger.info(
                    "{} completed ({})",
                    logPrefix(),
                    TextFormat.shortDebugString(result));

                afterComplete(result);
            });
    }

    private CompletableFuture<Void> reschedule(Status status) {
        var progress = progress(status);
        var attempt = attempt(progress);
        long delayMillis = randomize(backoff(BACKOFF_DELAY_MILLIS, BACKOFF_MAX_DELAY_MILLIS, attempt));
        long executeAt = currentTimeMillis() + delayMillis;
        return reschedule(executeAt, progress, status);
    }

    private CompletableFuture<Void> reschedule(long executeAt, P progress, @Nullable Status status) {
        actor.close();
        return context.reschedule(executeAt, progress)
            .thenRun(() -> {
                if (logger.isDebugEnabled()) {
                    logger.debug(
                        "{} rescheduled on {}, latest progress ({})",
                        logPrefix(),
                        Instant.ofEpochMilli(executeAt),
                        TextFormat.shortDebugString(progress));
                }

                afterReschedule(executeAt, progress, status);
            });
    }

    private P progress() {
        updateProgress();
        return progress.get();
    }

    private P progress(Status status) {
        updateProgress(status);
        return progress.get();
    }

    private boolean updateProgress() {
        return updateProgress(null);
    }

    private boolean updateProgress(@Nullable Status status) {
        P prev;
        P update;
        do {
            prev = progress.get();
            update = onUpdateProgress(prev, status);
            if (prev.equals(update)) {
                return false;
            }
        } while (!progress.compareAndSet(prev, update));

        if (logger.isDebugEnabled()) {
            logger.debug("{} progress ({})", logPrefix(), TextFormat.shortDebugString(update));
        }

        return true;
    }

    private void close(@Nullable Throwable t) {
        if (t != null && logger.isWarnEnabled()) {
            logger.warn("{} failed on complete", logPrefix(), t);
        }

        close();
    }

    private String logPrefix() {
        return type + "(task_id: \"" + context.task().id() + "\" " + TextFormat.shortDebugString(params) + ")";
    }

    private CompletableFuture<?> saveProgress(int attempt) {
        if (context.isDone()) {
            close();
            return completedFuture(null);
        }

        if (!updateProgress()) {
            return completedFuture(null);
        }

        return context.progress(progress.get())
            .whenComplete((ignore, e) -> {
                if (context.isDone()) {
                    close();
                }
            });
    }

}
