package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;

import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.solomon.core.conf.watch.SolomonConfHolder;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.coremon.tasks.AbstractTask;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.scheduler.Permit;
import ru.yandex.solomon.scheduler.PermitLimiter;
import ru.yandex.solomon.scheduler.TaskHandler;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.stockpile.client.StockpileClient;

import static java.lang.System.currentTimeMillis;
import static java.util.concurrent.TimeUnit.MINUTES;
import static java.util.concurrent.TimeUnit.SECONDS;
import static ru.yandex.misc.concurrent.CompletableFutures.safeCall;
import static ru.yandex.solomon.util.time.DurationUtils.randomize;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
abstract class AbstractDeleteMetricsTaskHandler<T extends AbstractTask<?>> implements TaskHandler {
    static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
        .withNumRetries(10)
        .withDelay(SECONDS.toMillis(1))
        .withMaxDelay(MINUTES.toMillis(1));

    private final PermitLimiter permitLimiter = new PermitLimiter(5);

    // to avoid intersections between parallel operations
    // restrict it to only 1 in-flight task per shard
    private final ConcurrentMap<Integer, String> inFlightShards = new ConcurrentHashMap<>();

    // for manager ui
    private final ConcurrentMap<String, T> running = new ConcurrentHashMap<>();

    final DeleteMetricsTaskMetrics metrics;
    final SolomonConfHolder confHolder;
    final StockpileClient stockpileClient;
    final MetabaseShardResolver<? extends MetabaseShard> shardResolver;

    AbstractDeleteMetricsTaskHandler(
        DeleteMetricsTaskMetrics metrics,
        SolomonConfHolder confHolder,
        StockpileClient stockpileClient,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver)
    {
        this.metrics = metrics;
        this.confHolder = confHolder;
        this.stockpileClient = stockpileClient;
        this.shardResolver = shardResolver;
    }

    @Override
    @Nullable
    public final Permit acquire(String id, Any params) {
        var conf = confHolder.getConf();
        if (conf == null) {
            return null;
        }

        if (stockpileClient.getAvailability().getAvailability() < 1.0) {
            return null;
        }

        if (shardResolver.isLoading()) {
            return null;
        }

        var numId = DeleteMetricsTaskProto.params(params).getNumId();
        var shard = shardResolver.resolveShardOrNull(numId);
        if (shard == null) {
            // if the shard does not exist globally,
            // execute on any node to be able to finish the job
            return conf.getShardByNumIdOrNull(numId) == null
                ? permitLimiter.acquire()
                : null;
        }

        if (!shard.isLoaded()) {
            return null;
        }

        if (inFlightShards.containsKey(numId)) {
            return null;
        }

        return permitLimiter.acquire();
    }

    @Override
    public final void execute(ExecutionContext context) {
        var taskId = context.task().id();

        var params = DeleteMetricsTaskProto.params(context.task().params());
        var numId = params.getNumId();

        var shardConf = confHolder.getConfOrThrow().getShardByNumIdOrNull(numId);
        if (shardConf != null) {
            metrics.touch(
                shardConf.getRaw().getProjectId(),
                shardConf.getId());
        }

        if (inFlightShards.putIfAbsent(numId, taskId) != null) {
            var rescheduleAt = currentTimeMillis() + randomize(MINUTES.toMillis(10));
            context.reschedule(rescheduleAt, context.task().progress());
            return;
        }

        safeCall(() -> {
            var task = createTask(context, params);
            running.put(taskId, task);
            return task.start().whenComplete((i, t) -> {
                running.remove(taskId, task);
                task.close();
            });
        }).whenComplete(
            (i, t) -> inFlightShards.remove(numId, taskId)
        );
    }

    abstract T createTask(ExecutionContext context, DeleteMetricsParams params);
}
