package ru.yandex.solomon.coremon.tasks.deleteMetrics;

import java.time.Duration;
import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import com.google.protobuf.TextFormat;
import io.grpc.Status;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.longs.LongSet;

import ru.yandex.coremon.api.task.DeleteMetricsCheckProgress.CheckNoRecentWritesProgress;
import ru.yandex.coremon.api.task.DeleteMetricsParams;
import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.service.MetabaseShard;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolver;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.collection.CloseableIterator;
import ru.yandex.solomon.util.future.RetryConfig;
import ru.yandex.solomon.util.future.RetryContext;
import ru.yandex.stockpile.api.EStockpileStatusCode;
import ru.yandex.stockpile.api.MetricMeta;
import ru.yandex.stockpile.api.ReadMetricsMetaRequest;
import ru.yandex.stockpile.client.StockpileClient;

import static java.lang.Math.toIntExact;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.solomon.coremon.meta.CoremonMetric.UNKNOWN_LAST_POINT_SECONDS;
import static ru.yandex.solomon.util.time.InstantUtils.millisecondsToSeconds;

/**
 * @author Stanislav Kashirin
 */
@ParametersAreNonnullByDefault
final class CheckNoRecentWrites implements AutoCloseable {

    //TODO: configure maxAsyncOperations? <SOLOMON-990>
    private static final int DEFAULT_MAX_ASYNC_OPERATIONS = 4;
    private static final int DEFAULT_MAX_BATCH_SIZE = 10_000;

    private static final Duration RECENT_WRITE_THRESHOLD = Duration.ofHours(1);

    private final RetryContext retryCtx;

    private final StockpileClient stockpileClient;
    private final MetabaseShardResolver<? extends MetabaseShard> shardResolver;
    private final Executor executor;

    private final DeleteMetricsParams params;
    private final Selectors selectors;

    private final AtomicReference<CheckNoRecentWritesProgress> progress;

    //TODO: upper bound for total nr of metrics in all incomplete batches <SOLOMON-990>
    private final int maxBatchSize;

    @Nullable
    private CheckActor checker;

    CheckNoRecentWrites(
        RetryConfig retry,
        StockpileClient stockpileClient,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        Executor executor,
        DeleteMetricsParams params,
        Selectors selectors,
        CheckNoRecentWritesProgress progress)
    {
        this(
            retry,
            stockpileClient,
            shardResolver,
            executor,
            params,
            selectors,
            progress,
            DEFAULT_MAX_BATCH_SIZE);
    }

    @VisibleForTesting
    CheckNoRecentWrites(
        RetryConfig retry,
        StockpileClient stockpileClient,
        MetabaseShardResolver<? extends MetabaseShard> shardResolver,
        Executor executor,
        DeleteMetricsParams params,
        Selectors selectors,
        CheckNoRecentWritesProgress progress,
        int maxBatchSize)
    {
        this.retryCtx = new RetryContext(retry);
        this.stockpileClient = stockpileClient;
        this.shardResolver = shardResolver;
        this.executor = executor;
        this.params = params;
        this.selectors = selectors;
        this.progress = new AtomicReference<>(progress);
        this.maxBatchSize = maxBatchSize;
    }

    public CompletableFuture<Void> start() {
        try {
            return tryStart();
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    public CheckNoRecentWritesProgress progress() {
        return progress.get();
    }

    private CompletableFuture<Void> tryStart() {
        if (progress().getComplete()) {
            return completedFuture(null);
        }

        var shard = shardResolver.resolveShardOrNull(params.getNumId());
        if (shard == null || !shard.isLoaded()) {
            return completedFuture(null);
        }

        var fileMetrics = shard.getStorage().getFileMetrics();
        var it = fileMetrics.searchIterator(selectors).iterator();

        checker = new CheckActor(it);

        return checker.start();
    }

    private CompletableFuture<LoadMetaBatch> loadMetaFromStockpile(LoadMetaBatch batch) {
        try {
            return readMetricsMeta(toReadMetricsMetaRequest(batch))
                .thenApply(metas -> {
                    for (var meta : metas) {
                        // not found in stockpile, keep it as unknown
                        if (meta.getLastTsMillis() == -1) {
                            continue;
                        }

                        var metric = batch.get(meta.getLocalId());
                        if (metric != null) {
                            metric.setLastPointSeconds(millisecondsToSeconds(meta.getLastTsMillis()));
                        }
                    }
                    return batch;
                });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private CompletableFuture<List<MetricMeta>> readMetricsMeta(ReadMetricsMetaRequest request) {
        return retry(
            () -> stockpileClient.readMetricsMeta(request)
                .thenApply(response -> {
                    if (response.getStatus() != EStockpileStatusCode.OK) {
                        var desc = "Unable to read metrics meta for " + TextFormat.shortDebugString(params) +
                            " from stockpile shard " + request.getShardId() +
                            " caused by " + response.getStatus() +
                            ": " + response.getStatusMessage();
                        throw new StatusRuntimeExceptionNoStackTrace(Status.ABORTED.withDescription(desc));
                    }

                    return response.getMetaList();
                })
        );
    }

    private <T> CompletableFuture<T> retry(Supplier<CompletableFuture<T>> supplier) {
        return retryCtx.retry(supplier);
    }

    private void complete(String recentWriteLabels) {
        complete(builder -> builder.setRecentWriteLabels(recentWriteLabels));
    }

    private void complete(int minLastPointSeconds) {
        complete(builder -> builder.setMinLastPointSeconds(minLastPointSeconds));
    }

    private void complete(Consumer<CheckNoRecentWritesProgress.Builder> updater) {
        var prev = progress.get();
        if (prev.getComplete()) {
            return;
        }

        var update = CheckNoRecentWritesProgress.newBuilder();
        updater.accept(update);

        progress.compareAndSet(prev, update.setComplete(true).build());
    }

    @Override
    public void close() {
        if (checker != null) {
            checker.close();
        }
        retryCtx.close();
    }

    private static ReadMetricsMetaRequest toReadMetricsMetaRequest(LoadMetaBatch batch) {
        var request = ReadMetricsMetaRequest.newBuilder()
            .setShardId(batch.getStockpileShardId());

        var it = batch.localIds().iterator();
        while (it.hasNext()) {
            request.addLocalIds(it.nextLong());
        }

        return request.build();
    }

    @ParametersAreNonnullByDefault
    private class CheckActor implements AutoCloseable {

        private final Int2ObjectMap<LoadMetaBatch> batchByStockpileShardId = new Int2ObjectOpenHashMap<>();

        private final CloseableIterator<? extends CoremonMetric> metricIt;
        private final int createdAtThresholdSeconds;
        private final int recentWriteThresholdSeconds;
        private final AsyncActorRunner actorRunner;

        private final CompletableFuture<Void> resultFuture = new CompletableFuture<>();

        private final AtomicInteger minLastPointSeconds = new AtomicInteger(Integer.MAX_VALUE);
        private int localMinLastPointSeconds = Integer.MAX_VALUE;

        private volatile boolean closed;

        CheckActor(CloseableIterator<? extends CoremonMetric> metricIt) {
            this.metricIt = metricIt;
            this.createdAtThresholdSeconds = millisecondsToSeconds(params.getCreatedAt());
            this.recentWriteThresholdSeconds = toIntExact(Instant.now().minus(RECENT_WRITE_THRESHOLD).getEpochSecond());
            this.actorRunner = new AsyncActorRunner(this::body, executor, DEFAULT_MAX_ASYNC_OPERATIONS);
        }

        CompletableFuture<Void> start() {
            var actorFuture = actorRunner.start()
                .whenComplete((i, t) -> cleanUp());
            CompletableFutures.whenComplete(actorFuture, resultFuture);
            return resultFuture.whenComplete((i, t) -> {
                if (t == null) {
                    complete(minLastPointSeconds.get());
                }
            });
        }

        private CompletableFuture<?> body() {
            if (closed) {
                return failedFuture(Status.CANCELLED.asRuntimeException());
            }

            var checkIteratorFuture = checkIterator();
            updateMinLastPointSeconds(localMinLastPointSeconds);
            if (checkIteratorFuture != null) {
                return checkIteratorFuture;
            }

            var loadNextBatchFuture = loadNextBatchIfExists();
            if (loadNextBatchFuture != null) {
                return loadNextBatchFuture;
            }

            return completedFuture(AsyncActorBody.DONE_MARKER);
        }

        @Nullable
        private CompletableFuture<?> checkIterator() {
            while (metricIt.hasNext()) {
                try (var metric = metricIt.next()) {
                    if (metric.getCreatedAtSeconds() > createdAtThresholdSeconds) {
                        continue;
                    }

                    var lastPointSeconds = metric.getLastPointSeconds();
                    if (lastPointSeconds == UNKNOWN_LAST_POINT_SECONDS) {
                        var spShardId = metric.getShardId();

                        var batch = batchByStockpileShardId.computeIfAbsent(spShardId, LoadMetaBatch::new);
                        batch.add(new FileCoremonMetric(metric));

                        if (batch.size() >= maxBatchSize) {
                            return loadMeta(batchByStockpileShardId.remove(spShardId));
                        }
                    } else {
                        localMinLastPointSeconds = Math.min(localMinLastPointSeconds, lastPointSeconds);
                        if (lastPointSeconds > recentWriteThresholdSeconds) {
                            return doneWithActiveMetricFound(metric);
                        }
                    }
                }
            }

            return null;
        }

        @Nullable
        private CompletableFuture<?> loadNextBatchIfExists() {
            var it = batchByStockpileShardId.values().iterator();
            if (it.hasNext()) {
                var batch = it.next();
                it.remove();
                return loadMeta(batch);
            }
            return null;
        }

        private CompletableFuture<?> loadMeta(LoadMetaBatch batch) {
            return loadMetaFromStockpile(batch)
                .thenCompose(this::checkLoadedMeta)
                .handle((i, t) -> {
                    if (t != null) {
                        resultFuture.completeExceptionally(t);
                        return AsyncActorBody.DONE_MARKER;
                    }
                    return null;
                });
        }

        private CompletableFuture<?> checkLoadedMeta(LoadMetaBatch batch) {
            var batchMinLastPointSeconds = Integer.MAX_VALUE;
            for (var metric : batch.metrics()) {
                var lastPointSeconds = metric.getLastPointSeconds();
                if (lastPointSeconds == UNKNOWN_LAST_POINT_SECONDS) {
                    // still unknown? stockpile knows nothing about this metric, so we may delete it
                    continue;
                }

                batchMinLastPointSeconds = Math.min(batchMinLastPointSeconds, lastPointSeconds);
                if (lastPointSeconds > recentWriteThresholdSeconds) {
                    return doneWithActiveMetricFound(metric);
                }
            }

            updateMinLastPointSeconds(batchMinLastPointSeconds);
            return completedFuture(null);
        }

        private void updateMinLastPointSeconds(int update) {
            int prev;
            do {
                prev = minLastPointSeconds.get();
                if (prev <= update) {
                    return;
                }

            } while (!minLastPointSeconds.compareAndSet(prev, update));
        }

        private CompletableFuture<?> doneWithActiveMetricFound(CoremonMetric metric) {
            complete(LabelsFormat.format(metric.getLabels()));
            return completedFuture(AsyncActorBody.DONE_MARKER);
        }

        private void cleanUp() {
            batchByStockpileShardId.clear();
            metricIt.close();
        }

        @Override
        public void close() {
            closed = true;
        }
    }

    @ParametersAreNonnullByDefault
    private static class LoadMetaBatch {

        private final Long2ObjectMap<FileCoremonMetric> metricByLocalId = new Long2ObjectOpenHashMap<>();

        private final int stockpileShardId;

        LoadMetaBatch(int stockpileShardId) {
            this.stockpileShardId = stockpileShardId;
        }

        int getStockpileShardId() {
            return stockpileShardId;
        }

        void add(FileCoremonMetric metric) {
            metricByLocalId.put(metric.getLocalId(), metric);
        }

        @Nullable
        FileCoremonMetric get(long localId) {
            return metricByLocalId.get(localId);
        }

        int size() {
            return metricByLocalId.size();
        }

        LongSet localIds() {
            return metricByLocalId.keySet();
        }

        Collection<FileCoremonMetric> metrics() {
            return metricByLocalId.values();
        }
    }
}
